"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fieldSerializer = exports.fieldDeserializer = void 0;
var _field_parameters = require("../components/document_fields/field_parameters");
var _constants = require("../constants");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const sanitizeField = field => Object.entries(field)
// If a parameter value is "index_default", we remove it
.filter(({
  1: value
}) => value !== _constants.INDEX_DEFAULT).reduce((acc, [param, value]) => ({
  ...acc,
  [param]: value
}), {});

/**
 * Run custom parameter serializers on field.
 * Each serializer takes the field as single argument and returns it serialized in an immutable way.
 * @param field The field we are serializing
 */
const runParametersSerializers = field => _field_parameters.PARAMETER_SERIALIZERS.reduce((fieldSerialized, serializer) => serializer(fieldSerialized), field);

/**
 * Run custom parameter deserializers on field.
 * Each deserializer takes the field as single argument and returns it deserialized in an immutable way.
 * @param field The field we are deserializing
 */
const runParametersDeserializers = field => _field_parameters.PARAMETER_DESERIALIZERS.reduce((fieldDeserialized, serializer) => serializer(fieldDeserialized), field);
const fieldSerializer = field => {
  const {
    otherTypeJson,
    ...rest
  } = field;
  const updatedField = Boolean(otherTypeJson) ? {
    ...otherTypeJson,
    ...rest
  } : {
    ...rest
  };

  // If a subType is present, use it as an Elasticsearch datatype
  if ({}.hasOwnProperty.call(updatedField, 'subType')) {
    updatedField.type = updatedField.subType;
    delete updatedField.subType;
  }

  // Delete temp fields
  delete updatedField.useSameAnalyzerForSearch;
  return sanitizeField(runParametersSerializers(updatedField));
};
exports.fieldSerializer = fieldSerializer;
const fieldDeserializer = field => {
  // Read the type provided and detect if it is a SubType or not
  // e.g. if "float" is provided, the field.type will be "numeric" and the field.subType will be "float"
  const typeMeta = (0, _utils.getTypeMetaFromSource)(field.type);
  field.type = typeMeta.mainType;
  field.subType = typeMeta.subType;
  if (field.type === 'other') {
    const {
      type,
      subType,
      name,
      ...otherTypeJson
    } = field;
    /**
     * For "other" type (type we don't support through a form)
     * we grab all the parameters and put them in the "otherTypeJson" object
     * that we will render in a JSON editor.
     */
    field.otherTypeJson = otherTypeJson;
  } else {
    field.useSameAnalyzerForSearch = {}.hasOwnProperty.call(field, 'search_analyzer') === false;
  }
  return runParametersDeserializers(field);
};
exports.fieldDeserializer = fieldDeserializer;