"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createArchiveIteratorFromMap = exports.createArchiveIterator = void 0;
var _ = require(".");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Creates an iterator for traversing and extracting paths from an archive
 * buffer. This iterator is intended to be used for memory efficient traversal
 * of archive contents without extracting the entire archive into memory.
 *
 * @param archiveBuffer - The buffer containing the archive data.
 * @param contentType - The content type of the archive (e.g.,
 * 'application/zip').
 * @returns ArchiveIterator instance.
 *
 */
const createArchiveIterator = (archiveBuffer, contentType) => {
  const paths = [];
  const traverseEntries = async (onEntry, readBuffer) => {
    await (0, _.traverseArchiveEntries)(archiveBuffer, contentType, async entry => {
      await onEntry(entry);
    }, readBuffer);
  };
  const getPaths = async () => {
    if (paths.length) {
      return paths;
    }
    await traverseEntries(async entry => {
      paths.push(entry.path);
    }, () => false);
    return paths;
  };
  return {
    traverseEntries,
    getPaths
  };
};

/**
 * Creates an archive iterator from the assetsMap. This is a stop-gap solution
 * to provide a uniform interface for traversing assets while assetsMap is still
 * in use. It works with a map of assets loaded into memory and is not intended
 * for use with large archives.
 *
 * @param assetsMap - A map where the keys are asset paths and the values are
 * asset buffers.
 * @returns ArchiveIterator instance.
 *
 */
exports.createArchiveIterator = createArchiveIterator;
const createArchiveIteratorFromMap = assetsMap => {
  const traverseEntries = async onEntry => {
    for (const [path, buffer] of assetsMap) {
      await onEntry({
        path,
        buffer
      });
    }
  };
  const getPaths = async () => {
    return [...assetsMap.keys()];
  };
  return {
    traverseEntries,
    getPaths
  };
};
exports.createArchiveIteratorFromMap = createArchiveIteratorFromMap;