"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports._updatePackagePoliciesThatNeedBump = _updatePackagePoliciesThatNeedBump;
exports.registerBumpAgentPoliciesTask = registerBumpAgentPoliciesTask;
exports.scheduleBumpAgentPoliciesTask = scheduleBumpAgentPoliciesTask;
var _uuid = require("uuid");
var _lodash = require("lodash");
var _common = require("@kbn/spaces-plugin/common");
var _ = require("..");
var _cache = require("../epm/packages/cache");
var _package_policy = require("../package_policy");
var _package_policies = require("../package_policies");
var _saved_object = require("../saved_object");
var _constants = require("../../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TASK_TYPE = 'fleet:bump_agent_policies';
function registerBumpAgentPoliciesTask(taskManagerSetup) {
  taskManagerSetup.registerTaskDefinitions({
    [TASK_TYPE]: {
      title: 'Fleet Bump policies',
      timeout: '5m',
      maxAttempts: 3,
      createTaskRunner: ({
        taskInstance
      }) => {
        let cancelled = false;
        const isCancelled = () => cancelled;
        return {
          async run() {
            if (isCancelled()) {
              throw new Error('Task has been cancelled');
            }
            await (0, _cache.runWithCache)(async () => {
              await _updatePackagePoliciesThatNeedBump(_.appContextService.getLogger(), isCancelled);
            });
          },
          async cancel() {
            cancelled = true;
          }
        };
      }
    }
  });
}
async function getPackagePoliciesToBump(savedObjectType) {
  const result = await _.appContextService.getInternalUserSOClientWithoutSpaceExtension().find({
    type: savedObjectType,
    filter: (0, _saved_object.normalizeKuery)(savedObjectType, `${savedObjectType}.bump_agent_policy_revision:true`),
    perPage: _constants.SO_SEARCH_LIMIT,
    namespaces: ['*'],
    fields: ['id', 'namespaces', 'policy_ids']
  });
  return {
    total: result.total,
    items: result.saved_objects.map(so => (0, _package_policies.mapPackagePolicySavedObjectToPackagePolicy)(so, so.namespaces))
  };
}
async function _updatePackagePoliciesThatNeedBump(logger, isCancelled) {
  const savedObjectType = await (0, _package_policy.getPackagePolicySavedObjectType)();
  const packagePoliciesToBump = await getPackagePoliciesToBump(savedObjectType);
  logger.info(`Found ${packagePoliciesToBump.total} package policies that need agent policy revision bump`);
  const packagePoliciesIndexedBySpace = packagePoliciesToBump.items.reduce((acc, policy) => {
    var _policy$spaceIds$, _policy$spaceIds;
    const spaceId = (_policy$spaceIds$ = (_policy$spaceIds = policy.spaceIds) === null || _policy$spaceIds === void 0 ? void 0 : _policy$spaceIds[0]) !== null && _policy$spaceIds$ !== void 0 ? _policy$spaceIds$ : _common.DEFAULT_SPACE_ID;
    if (!acc[spaceId]) {
      acc[spaceId] = [];
    }
    acc[spaceId].push(policy);
    return acc;
  }, {});
  const start = Date.now();
  for (const [spaceId, packagePolicies] of Object.entries(packagePoliciesIndexedBySpace)) {
    if (isCancelled()) {
      throw new Error('Task has been cancelled');
    }
    const soClient = _.appContextService.getInternalUserSOClientForSpaceId(spaceId);
    const esClient = _.appContextService.getInternalUserESClient();
    await soClient.bulkUpdate(packagePolicies.map(item => ({
      type: savedObjectType,
      id: item.id,
      attributes: {
        bump_agent_policy_revision: false
      }
    })));
    const updatedCount = packagePolicies.length;
    const agentPoliciesToBump = (0, _lodash.uniq)(packagePolicies.map(item => item.policy_ids).flat());
    await _.agentPolicyService.bumpAgentPoliciesByIds(soClient, esClient, agentPoliciesToBump);
    logger.debug(`Updated ${updatedCount} package policies in space ${spaceId} in ${Date.now() - start}ms, bump ${agentPoliciesToBump.length} agent policies`);
  }
}
async function scheduleBumpAgentPoliciesTask(taskManagerStart) {
  await taskManagerStart.ensureScheduled({
    id: `${TASK_TYPE}:${(0, _uuid.v4)()}`,
    scope: ['fleet'],
    params: {},
    taskType: TASK_TYPE,
    runAt: new Date(Date.now() + 3 * 1000),
    state: {}
  });
}