"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useComboBoxWithCustomInput = useComboBoxWithCustomInput;
exports.useComboInput = useComboInput;
exports.useInput = useInput;
exports.useKeyValueInput = useKeyValueInput;
exports.useNumberInput = useNumberInput;
exports.useRadioInput = useRadioInput;
exports.useSecretInput = useSecretInput;
exports.useSelectInput = useSelectInput;
exports.useSwitchInput = useSwitchInput;
exports.validateInputs = validateInputs;
var _react = require("react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function validateInputs(inputs) {
  return Object.values(inputs).reduce((acc, input) => {
    const res = input.validate();
    return acc === false ? acc : res;
  }, true);
}
function useInput(defaultValue = '', validate, disabled = false) {
  const [value, setValue] = (0, _react.useState)(defaultValue);
  const [errors, setErrors] = (0, _react.useState)();
  const [hasChanged, setHasChanged] = (0, _react.useState)(false);
  const onChange = (0, _react.useCallback)(e => {
    const newValue = e.target.value;
    setValue(newValue);
    if (errors && validate && validate(newValue) === undefined) {
      setErrors(undefined);
    }
  }, [errors, validate]);
  (0, _react.useEffect)(() => {
    if (hasChanged) {
      return;
    }
    if (value !== defaultValue) {
      setHasChanged(true);
    }
  }, [hasChanged, value, defaultValue]);
  const isInvalid = errors !== undefined;
  return {
    value,
    errors,
    props: {
      onChange,
      value,
      isInvalid,
      disabled
    },
    formRowProps: {
      error: errors,
      isInvalid
    },
    clear: () => {
      setValue('');
    },
    validate: () => {
      if (validate) {
        const newErrors = validate(value);
        setErrors(newErrors);
        return newErrors === undefined;
      }
      return true;
    },
    setValue,
    hasChanged
  };
}
function useSecretInput(initialValue, validate, disabled = false) {
  const [value, setValue] = (0, _react.useState)(initialValue);
  const [errors, setErrors] = (0, _react.useState)();
  const [hasChanged, setHasChanged] = (0, _react.useState)(false);
  const onChange = (0, _react.useCallback)(e => {
    const newValue = e.target.value;
    setValue(newValue);
    if (errors && validate && validate(newValue) === undefined) {
      setErrors(undefined);
    }
  }, [errors, validate]);
  (0, _react.useEffect)(() => {
    if (hasChanged) {
      return;
    }
    if (value !== initialValue) {
      setHasChanged(true);
    }
  }, [hasChanged, value, initialValue]);
  const isInvalid = errors !== undefined;
  return {
    value,
    errors,
    props: {
      onChange,
      value: typeof value === 'string' ? value : '',
      isInvalid,
      disabled
    },
    formRowProps: {
      error: errors,
      isInvalid,
      initialValue,
      disabled,
      clear: () => {
        setValue('');
      }
    },
    cancelEdit: () => {
      setValue(initialValue || '');
    },
    validate: () => {
      if (validate) {
        const newErrors = validate(value);
        setErrors(newErrors);
        return newErrors === undefined;
      }
      return true;
    },
    setValue,
    hasChanged
  };
}
function useRadioInput(defaultValue, disabled = false) {
  const [value, setValue] = (0, _react.useState)(defaultValue);
  const [hasChanged, setHasChanged] = (0, _react.useState)(false);
  (0, _react.useEffect)(() => {
    if (hasChanged) {
      return;
    }
    if (value !== defaultValue) {
      setHasChanged(true);
    }
  }, [hasChanged, value, defaultValue]);
  const onChange = (0, _react.useCallback)(setValue, [setValue]);
  return {
    props: {
      idSelected: value,
      onChange,
      disabled
    },
    setValue,
    value,
    hasChanged
  };
}
function useSwitchInput(defaultValue = false, disabled = false) {
  const [value, setValue] = (0, _react.useState)(defaultValue);
  const [hasChanged, setHasChanged] = (0, _react.useState)(false);
  (0, _react.useEffect)(() => {
    if (hasChanged) {
      return;
    }
    if (value !== defaultValue) {
      setHasChanged(true);
    }
  }, [hasChanged, value, defaultValue]);
  const onChange = e => {
    const newValue = e.target.checked;
    setValue(newValue);
  };
  const validate = (0, _react.useCallback)(() => true, []);
  return {
    value,
    props: {
      onChange,
      checked: value,
      disabled
    },
    validate,
    formRowProps: {},
    setValue,
    hasChanged
  };
}
function useCustomInput(id, defaultValue, validate, disabled = false) {
  const [value, setValue] = (0, _react.useState)(defaultValue);
  const [errors, setErrors] = (0, _react.useState)();
  const [hasChanged, setHasChanged] = (0, _react.useState)(false);
  (0, _react.useEffect)(() => {
    if (hasChanged) {
      return;
    }
    if (JSON.stringify(value) !== JSON.stringify(defaultValue)) {
      setHasChanged(true);
    }
  }, [hasChanged, value, defaultValue]);
  const isInvalid = errors !== undefined;
  const validateCallback = (0, _react.useCallback)(() => {
    if (validate) {
      const newErrors = validate(value);
      setErrors(newErrors);
      return newErrors === undefined;
    }
    return true;
  }, [validate, value]);
  const onChange = (0, _react.useCallback)(newValue => {
    setValue(newValue);
    if (errors && validate) {
      setErrors(validate(newValue));
    }
  }, [validate, errors]);
  return {
    props: {
      id,
      value,
      onChange,
      errors,
      isInvalid,
      disabled
    },
    formRowProps: {
      error: errors,
      isInvalid
    },
    value,
    clear: () => {
      setValue(defaultValue);
    },
    setValue,
    validate: validateCallback,
    hasChanged
  };
}
function useComboInput(id, defaultValue = [], validate, disabled = false) {
  return useCustomInput(id, defaultValue, validate, disabled);
}
function useKeyValueInput(id, defaultValue = [], validate, disabled = false) {
  return useCustomInput(id, defaultValue, validate, disabled);
}
function useNumberInput(defaultValue, validate, disabled = false) {
  const [value, setValue] = (0, _react.useState)(defaultValue);
  const [errors, setErrors] = (0, _react.useState)();
  const [hasChanged, setHasChanged] = (0, _react.useState)(false);
  const onChange = (0, _react.useCallback)(e => {
    const newValue = e.target.value ? Number(e.target.value) : undefined;
    setValue(newValue);
    if (newValue && errors && validate && validate(newValue) === undefined) {
      setErrors(undefined);
    }
  }, [errors, validate]);
  (0, _react.useEffect)(() => {
    if (hasChanged) {
      return;
    }
    if (value !== defaultValue) {
      setHasChanged(true);
    }
  }, [hasChanged, value, defaultValue]);
  const isInvalid = errors !== undefined;
  return {
    value,
    errors,
    props: {
      onChange,
      value,
      isInvalid,
      disabled
    },
    formRowProps: {
      error: errors,
      isInvalid
    },
    clear: () => {
      setValue(undefined);
    },
    validate: () => {
      if (validate && value) {
        const newErrors = validate(value);
        setErrors(newErrors);
        return newErrors === undefined;
      }
      return true;
    },
    setValue,
    hasChanged
  };
}
function useSelectInput(options, defaultValue = '', disabled = false) {
  const [value, setValue] = (0, _react.useState)(defaultValue);
  const onChange = (0, _react.useCallback)(e => {
    setValue(e.target.value);
  }, []);
  return {
    props: {
      options,
      value,
      onChange,
      disabled
    },
    value,
    clear: () => {
      setValue('');
    },
    setValue
  };
}
function useComboBoxWithCustomInput(id, defaultValue = [], validate, disabled = false) {
  var _selectedOptions$;
  const [selectedOptions, setSelected] = (0, _react.useState)(defaultValue);
  const [errors, setErrors] = (0, _react.useState)();
  const onChange = (0, _react.useCallback)(selected => {
    setSelected(selected);
    if (errors && validate) {
      setErrors(validate(selected));
    }
  }, [validate, errors]);
  const onCreateOption = searchValue => {
    const normalizedSearchValue = searchValue.trim();
    const newOption = {
      label: normalizedSearchValue,
      value: normalizedSearchValue
    };
    setSelected([newOption]);
    if (validate) {
      setErrors(validate([newOption]));
    }
  };
  const validateCallback = (0, _react.useCallback)(() => {
    if (validate) {
      const newErrors = validate(selectedOptions);
      setErrors(newErrors);
      return newErrors === undefined;
    }
    return true;
  }, [validate, selectedOptions]);
  const value = selectedOptions.length > 0 ? (_selectedOptions$ = selectedOptions[0]) === null || _selectedOptions$ === void 0 ? void 0 : _selectedOptions$.value : '';
  const isInvalid = errors !== undefined;
  return {
    props: {
      id,
      onChange,
      onCreateOption,
      errors,
      isInvalid,
      disabled,
      selectedOptions
    },
    formRowProps: {
      error: errors,
      isInvalid
    },
    value,
    clear: () => {
      setSelected(defaultValue);
    },
    setSelected,
    validate: validateCallback
  };
}