"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isSupportedDataType = exports.isReturnType = exports.isParameterType = exports.isFieldType = exports.getLocationFromCommandOrOptionName = exports.fieldTypes = exports.dataTypes = exports.Location = exports.FunctionDefinitionTypes = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * All supported field types in ES|QL. This is all the types
 * that can come back in the table from a query.
 */
const fieldTypes = exports.fieldTypes = ['boolean', 'date', 'double', 'ip', 'keyword', 'integer', 'long', 'text', 'unsigned_long', 'version', 'cartesian_point', 'cartesian_shape', 'geo_point', 'geo_shape', 'counter_integer', 'counter_long', 'counter_double', 'unsupported', 'date_nanos', 'function_named_parameters'];
const isFieldType = type => fieldTypes.includes(type);

/**
 * This is the list of all data types that are supported in ES|QL.
 *
 * Not all of these can be used as field types. Some can only be literals,
 * others may be the value of a field, but cannot be used in the index mapping.
 *
 * This is a partial list. The full list is here and we may need to expand this type as
 * the capabilities of the client-side engines grow.
 * https://github.com/elastic/elasticsearch/blob/main/x-pack/plugin/esql-core/src/main/java/org/elasticsearch/xpack/esql/core/type/DataType.java
 */
exports.isFieldType = isFieldType;
const dataTypes = exports.dataTypes = [...fieldTypes, 'null', 'time_duration', 'date_period', 'param' // Defines a named param such as ?value or ??field
];
const isSupportedDataType = type => dataTypes.includes(type);

/**
 * This is a set of array types. These aren't official ES|QL types, but they are
 * currently used in the function definitions in a couple of specific scenarios.
 *
 * The fate of these is uncertain. They may be removed in the future.
 */
exports.isSupportedDataType = isSupportedDataType;
const arrayTypes = ['double[]', 'unsigned_long[]', 'long[]', 'integer[]', 'counter_integer[]', 'counter_long[]', 'counter_double[]', 'keyword[]', 'text[]', 'boolean[]', 'any[]', 'date[]', 'date_period[]', 'ip[]', 'cartesian_point[]', 'cartesian_shape[]', 'geo_point[]', 'geo_shape[]', 'version[]', 'date_nanos[]'];

/**
 * This is the type of a parameter in a function definition.
 */

const isParameterType = str => typeof str !== undefined && str !== 'unsupported' && [...dataTypes, ...arrayTypes, 'any'].includes(str);

/**
 * This is the return type of a function definition.
 *
 * TODO: remove `any`
 */
exports.isParameterType = isParameterType;
const isReturnType = str => str !== 'unsupported' && (dataTypes.includes(str) || str === 'unknown' || str === 'any');
exports.isReturnType = isReturnType;
let FunctionDefinitionTypes = exports.FunctionDefinitionTypes = /*#__PURE__*/function (FunctionDefinitionTypes) {
  FunctionDefinitionTypes["AGG"] = "agg";
  FunctionDefinitionTypes["SCALAR"] = "scalar";
  FunctionDefinitionTypes["OPERATOR"] = "operator";
  FunctionDefinitionTypes["GROUPING"] = "grouping";
  return FunctionDefinitionTypes;
}({});
/**
 * This is a list of locations within an ES|QL query.
 *
 * It is currently used to suggest appropriate functions and
 * operators given the location of the cursor.
 */
let Location = exports.Location = /*#__PURE__*/function (Location) {
  Location["EVAL"] = "eval";
  Location["WHERE"] = "where";
  Location["ROW"] = "row";
  Location["SORT"] = "sort";
  Location["STATS"] = "stats";
  Location["STATS_BY"] = "stats_by";
  Location["STATS_WHERE"] = "stats_where";
  Location["ENRICH"] = "enrich";
  Location["ENRICH_WITH"] = "enrich_with";
  Location["DISSECT"] = "dissect";
  Location["RENAME"] = "rename";
  Location["JOIN"] = "join";
  Location["SHOW"] = "show";
  Location["COMPLETION"] = "completion";
  return Location;
}({});
const commandOptionNameToLocation = {
  eval: Location.EVAL,
  where: Location.WHERE,
  row: Location.ROW,
  sort: Location.SORT,
  stats: Location.STATS,
  by: Location.STATS_BY,
  enrich: Location.ENRICH,
  with: Location.ENRICH_WITH,
  dissect: Location.DISSECT,
  rename: Location.RENAME,
  join: Location.JOIN,
  show: Location.SHOW,
  completion: Location.COMPLETION
};

/**
 * Pause before using this in new places. Where possible, use the Location enum directly.
 *
 * This is primarily around for backwards compatibility with the old system of command and option names.
 */
const getLocationFromCommandOrOptionName = name => commandOptionNameToLocation[name];
exports.getLocationFromCommandOrOptionName = getLocationFromCommandOrOptionName;