"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CompletionPosition = void 0;
exports.suggest = suggest;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _helpers = require("../../../shared/helpers");
var _factories = require("../../factories");
var _constants = require("../../../shared/constants");
var _complete_items = require("../../complete_items");
var _types = require("../../../definitions/types");
var _helper = require("../../helper");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
let CompletionPosition = exports.CompletionPosition = /*#__PURE__*/function (CompletionPosition) {
  CompletionPosition["AFTER_COMPLETION"] = "after_completion";
  CompletionPosition["AFTER_PROMPT_OR_TARGET"] = "after_prompt_or_target";
  CompletionPosition["AFTER_PROMPT"] = "after_prompt";
  CompletionPosition["AFTER_WITH"] = "after_with";
  CompletionPosition["AFTER_INFERENCE_ID"] = "after_inference_id";
  CompletionPosition["AFTER_TARGET_ID"] = "after_target_id";
  return CompletionPosition;
}({});
function getPosition(params) {
  const {
    innerText,
    getExpressionType
  } = params;
  const {
    prompt,
    inferenceId,
    targetField
  } = params.command;
  if (inferenceId.incomplete && /WITH\s*$/i.test(innerText)) {
    return CompletionPosition.AFTER_WITH;
  }
  if (!inferenceId.incomplete) {
    return CompletionPosition.AFTER_INFERENCE_ID;
  }
  const expressionRoot = (prompt === null || prompt === void 0 ? void 0 : prompt.text) !== _constants.EDITOR_MARKER ? prompt : undefined;
  const expressionType = getExpressionType(expressionRoot);
  if ((0, _helper.isExpressionComplete)(expressionType, innerText)) {
    return CompletionPosition.AFTER_PROMPT;
  }
  if (targetField && !targetField.incomplete) {
    return CompletionPosition.AFTER_TARGET_ID;
  }

  // If we are right after COMPLETION or if there is only one word with no space after it (for fragments).
  if (!(expressionRoot !== null && expressionRoot !== void 0 && expressionRoot.text) || /COMPLETION\s*\S*$/i.test(innerText)) {
    return CompletionPosition.AFTER_COMPLETION;
  }

  // We don't know if the expression is a prompt or a target field
  if (prompt.type === 'unknown' && prompt.name === 'unknown') {
    return CompletionPosition.AFTER_PROMPT_OR_TARGET;
  }
  return undefined;
}
async function suggest(params) {
  var _callbacks$getInferen, _result$inferenceEndp;
  const {
    references,
    innerText,
    columnExists,
    getColumnsByType,
    callbacks,
    getSuggestedUserDefinedColumnName
  } = params;
  const {
    prompt
  } = params.command;
  const position = getPosition(params);
  switch (position) {
    case CompletionPosition.AFTER_COMPLETION:
    case CompletionPosition.AFTER_TARGET_ID:
      {
        const fieldsAndFunctionsSuggestions = (0, _lodash.uniqBy)(await (0, _helper.getFieldsOrFunctionsSuggestions)(['text', 'keyword', 'unknown'], _types.Location.COMPLETION, getColumnsByType, {
          functions: true,
          fields: true,
          userDefinedColumns: references === null || references === void 0 ? void 0 : references.userDefinedColumns
        }), 'label');
        const suggestions = await (0, _helper.handleFragment)(innerText, fragment => Boolean(columnExists(fragment) || (0, _helpers.getFunctionDefinition)(fragment)), (_fragment, rangeToReplace) => {
          return fieldsAndFunctionsSuggestions.map(suggestion => {
            return {
              ...suggestion,
              text: `${suggestion.text} `,
              command: _factories.TRIGGER_SUGGESTION_COMMAND,
              rangeToReplace
            };
          });
        }, () => []);
        const lastWord = (0, _helpers.findFinalWord)(innerText);
        if (!lastWord) {
          suggestions.push(defaultPrompt);
        }
        if (position !== CompletionPosition.AFTER_TARGET_ID) {
          suggestions.push((0, _factories.getNewUserDefinedColumnSuggestion)(getSuggestedUserDefinedColumnName()));
        }
        return suggestions;
      }
    case CompletionPosition.AFTER_PROMPT:
      return [withCompletionItem];
    case CompletionPosition.AFTER_PROMPT_OR_TARGET:
      {
        const lastWord = (0, _helpers.findFinalWord)(innerText);
        if (!lastWord.length && !columnExists(prompt.text) && !prompt.text.startsWith(_constants.ESQL_VARIABLES_PREFIX)) {
          return [assignCompletionItem];
        }
        return [withCompletionItem];
      }
    case CompletionPosition.AFTER_WITH:
      const result = await (callbacks === null || callbacks === void 0 ? void 0 : (_callbacks$getInferen = callbacks.getInferenceEndpoints) === null || _callbacks$getInferen === void 0 ? void 0 : _callbacks$getInferen.call(callbacks, 'completion'));
      return (result === null || result === void 0 ? void 0 : (_result$inferenceEndp = result.inferenceEndpoints) === null || _result$inferenceEndp === void 0 ? void 0 : _result$inferenceEndp.map(inferenceEndpointToCompletionItem)) || [];
    case CompletionPosition.AFTER_INFERENCE_ID:
      return [_complete_items.pipeCompleteItem];
    default:
      return [];
  }
}
const defaultPrompt = {
  detail: '',
  kind: 'Constant',
  asSnippet: true,
  label: 'Your prompt to the LLM',
  sortText: '1',
  text: '"${0:Your prompt to the LLM.}"'
};
const withCompletionItem = {
  detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definitions.completionWithDoc', {
    defaultMessage: 'Provide additional parameters for the LLM prompt.'
  }),
  kind: 'Reference',
  label: 'WITH',
  sortText: '1',
  text: 'WITH ',
  command: _factories.TRIGGER_SUGGESTION_COMMAND
};
const assignCompletionItem = {
  detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.newVarDoc', {
    defaultMessage: 'Define a new column'
  }),
  command: _factories.TRIGGER_SUGGESTION_COMMAND,
  label: '=',
  kind: 'Variable',
  sortText: '1',
  text: '= '
};
function inferenceEndpointToCompletionItem(inferenceEndpoint) {
  return {
    detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definitions.completionInferenceIdDoc', {
      defaultMessage: 'Inference endpoint used for the completion'
    }),
    kind: 'Reference',
    label: inferenceEndpoint.inference_id,
    sortText: '1',
    text: `\`${inferenceEndpoint.inference_id}\` `,
    command: _factories.TRIGGER_SUGGESTION_COMMAND
  };
}