"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.disableEntityDiscoveryRoute = void 0;
var _zod = require("@kbn/zod");
var _zodHelpers = require("@kbn/zod-helpers");
var _auth = require("../../lib/auth");
var _privileges = require("../../lib/auth/privileges");
var _uninstall_entity_definition = require("../../lib/entities/uninstall_entity_definition");
var _saved_objects = require("../../saved_objects");
var _create_entity_manager_server_route = require("../create_entity_manager_server_route");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * @openapi
 * /internal/entities/managed/enablement:
 *   delete:
 *     description: Disable managed (built-in) entity discovery. This stops and deletes the transforms, ingest pipelines, definitions saved objects, and index templates for this entity definition, as well as the stored API key for entity discovery management.
 *     tags:
 *       - management
 *     parameters:
 *       - in: query
 *         name: deleteData
 *         description: If true, delete all entity data in the managed indices
 *         required: false
 *         schema:
 *           type: boolean
 *           default: false
 *     responses:
 *       200:
 *         description: Built-in entity discovery successfully disabled
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               required: success
 *               properties:
 *                 success:
 *                  type: boolean
 *       403:
 *         description: The current user does not have the required permissions to disable entity discovery
 */
const disableEntityDiscoveryRoute = exports.disableEntityDiscoveryRoute = (0, _create_entity_manager_server_route.createEntityManagerServerRoute)({
  endpoint: 'DELETE /internal/entities/managed/enablement',
  security: {
    authz: {
      enabled: false,
      reason: 'This endpoint leverages the security plugin to evaluate the privileges needed as part of its core flow'
    }
  },
  params: _zod.z.object({
    query: _zod.z.object({
      deleteData: _zod.z.optional(_zodHelpers.BooleanFromString).default(false)
    })
  }),
  handler: async ({
    context,
    request,
    response,
    params,
    logger,
    server,
    getScopedClient
  }) => {
    try {
      const esClientAsCurrentUser = (await context.core).elasticsearch.client.asCurrentUser;
      const canDisable = await (0, _privileges.canDisableEntityDiscovery)(esClientAsCurrentUser);
      if (!canDisable) {
        return response.forbidden({
          body: {
            message: 'Current Kibana user does not have the required permissions to disable entity discovery'
          }
        });
      }
      const entityClient = await getScopedClient({
        request
      });
      const soClient = (await context.core).savedObjects.getClient({
        includedHiddenTypes: [_saved_objects.EntityDiscoveryApiKeyType.name]
      });
      logger.info('Disabling managed entity discovery');
      await (0, _uninstall_entity_definition.uninstallBuiltInEntityDefinitions)({
        entityClient,
        deleteData: params.query.deleteData
      });
      logger.debug('reading entity discovery API key from saved object');
      const apiKey = await (0, _auth.readEntityDiscoveryAPIKey)(server);
      // api key could be deleted outside of the apis, it does not affect the
      // disablement flow
      if (apiKey) {
        await (0, _auth.deleteEntityDiscoveryAPIKey)(soClient);
        await server.security.authc.apiKeys.invalidateAsInternalUser({
          ids: [apiKey.id]
        });
      }
      logger.info('Managed entity discovery is disabled');
      return response.ok({
        body: {
          success: true
        }
      });
    } catch (err) {
      logger.error(err);
      return response.customError({
        statusCode: 500,
        body: err
      });
    }
  }
});