"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerConnectorRoutes = registerConnectorRoutes;
var _configSchema = require("@kbn/config-schema");
var _i18n = require("@kbn/i18n");
var _searchConnectors = require("@kbn/search-connectors");
var _cancel_syncs = require("@kbn/search-connectors/lib/cancel_syncs");
var _identify_exceptions = require("@kbn/search-connectors/utils/identify_exceptions");
var _error_codes = require("../../../common/types/error_codes");
var _add_connector = require("../../lib/connectors/add_connector");
var _generate_config = require("../../lib/connectors/generate_config");
var _generate_connector_name = require("../../lib/connectors/generate_connector_name");
var _start_sync = require("../../lib/connectors/start_sync");
var _delete_access_control_index = require("../../lib/indices/delete_access_control_index");
var _fetch_index_counts = require("../../lib/indices/fetch_index_counts");
var _fetch_unattached_indices = require("../../lib/indices/fetch_unattached_indices");
var _generate_api_key = require("../../lib/indices/generate_api_key");
var _delete_pipelines = require("../../lib/pipelines/delete_pipelines");
var _get_default_pipeline = require("../../lib/pipelines/get_default_pipeline");
var _update_default_pipeline = require("../../lib/pipelines/update_default_pipeline");
var _update_pipeline = require("../../lib/pipelines/update_pipeline");
var _create_error = require("../../utils/create_error");
var _elasticsearch_error_handler = require("../../utils/elasticsearch_error_handler");
var _identify_exceptions2 = require("../../utils/identify_exceptions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerConnectorRoutes({
  router,
  log
}) {
  router.post({
    path: '/internal/enterprise_search/connectors',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        delete_existing_connector: _configSchema.schema.maybe(_configSchema.schema.boolean()),
        index_name: _configSchema.schema.maybe(_configSchema.schema.string()),
        is_native: _configSchema.schema.boolean(),
        language: _configSchema.schema.nullable(_configSchema.schema.string()),
        name: _configSchema.schema.maybe(_configSchema.schema.string()),
        service_type: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    try {
      var _request$body$index_n, _request$body$name;
      const body = await (0, _add_connector.addConnector)(client, {
        deleteExistingConnector: request.body.delete_existing_connector,
        indexName: (_request$body$index_n = request.body.index_name) !== null && _request$body$index_n !== void 0 ? _request$body$index_n : null,
        isNative: request.body.is_native,
        language: request.body.language,
        name: (_request$body$name = request.body.name) !== null && _request$body$name !== void 0 ? _request$body$name : null,
        serviceType: request.body.service_type
      });
      return response.ok({
        body
      });
    } catch (error) {
      if (error.message === _error_codes.ErrorCode.CONNECTOR_DOCUMENT_ALREADY_EXISTS || error.message === _error_codes.ErrorCode.INDEX_ALREADY_EXISTS) {
        return (0, _create_error.createError)({
          errorCode: error.message,
          message: _i18n.i18n.translate('xpack.enterpriseSearch.server.routes.addConnector.connectorExistsError', {
            defaultMessage: 'Connector or index already exists'
          }),
          response,
          statusCode: 409
        });
      }
      throw error;
    }
  }));
  router.post({
    path: '/internal/enterprise_search/connectors/{connectorId}/cancel_syncs',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    await (0, _cancel_syncs.cancelSyncs)(client.asCurrentUser, request.params.connectorId);
    return response.ok();
  }));
  router.put({
    path: '/internal/enterprise_search/connectors/{syncJobId}/cancel_sync',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        syncJobId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    try {
      await (0, _searchConnectors.cancelSync)(client.asCurrentUser, request.params.syncJobId);
    } catch (error) {
      if ((0, _identify_exceptions.isStatusTransitionException)(error)) {
        return (0, _create_error.createError)({
          errorCode: _error_codes.ErrorCode.STATUS_TRANSITION_ERROR,
          message: _i18n.i18n.translate('xpack.enterpriseSearch.server.routes.connectors.statusTransitionError', {
            defaultMessage: 'Connector sync job cannot be cancelled. Connector is already cancelled or not in a cancelable state.'
          }),
          response,
          statusCode: 400
        });
      }
      throw error;
    }
    return response.ok();
  }));
  router.post({
    path: '/internal/enterprise_search/connectors/{connectorId}/configuration',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      body: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.number(), _configSchema.schema.boolean()])),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const configuration = await (0, _searchConnectors.updateConnectorConfiguration)(client.asCurrentUser, request.params.connectorId, request.body);
    return response.ok({
      body: configuration
    });
  }));
  router.post({
    path: '/internal/enterprise_search/connectors/{connectorId}/scheduling',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        access_control: _configSchema.schema.object({
          enabled: _configSchema.schema.boolean(),
          interval: _configSchema.schema.string()
        }),
        full: _configSchema.schema.object({
          enabled: _configSchema.schema.boolean(),
          interval: _configSchema.schema.string()
        }),
        incremental: _configSchema.schema.object({
          enabled: _configSchema.schema.boolean(),
          interval: _configSchema.schema.string()
        })
      }),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    await (0, _searchConnectors.updateConnectorScheduling)(client.asCurrentUser, request.params.connectorId, request.body);
    return response.ok();
  }));
  router.post({
    path: '/internal/enterprise_search/connectors/{connectorId}/start_sync',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        nextSyncConfig: _configSchema.schema.maybe(_configSchema.schema.string())
      }),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    await (0, _start_sync.startSync)(client, request.params.connectorId, _searchConnectors.SyncJobType.FULL, request.body.nextSyncConfig);
    return response.ok();
  }));
  router.post({
    path: '/internal/enterprise_search/connectors/{connectorId}/start_incremental_sync',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    await (0, _start_sync.startSync)(client, request.params.connectorId, _searchConnectors.SyncJobType.INCREMENTAL);
    return response.ok();
  }));
  router.post({
    path: '/internal/enterprise_search/connectors/{connectorId}/start_access_control_sync',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    try {
      const {
        client
      } = (await context.core).elasticsearch;
      await (0, _start_sync.startSync)(client, request.params.connectorId, _searchConnectors.SyncJobType.ACCESS_CONTROL);
      return response.ok();
    } catch (error) {
      if ((0, _identify_exceptions2.isAccessControlDisabledException)(error)) {
        return (0, _create_error.createError)({
          errorCode: _error_codes.ErrorCode.ACCESS_CONTROL_DISABLED,
          message: _i18n.i18n.translate('xpack.enterpriseSearch.server.connectors.accessControlSync.accessControlDisabledError', {
            defaultMessage: 'Access control sync cannot be created. You must first enable Document Level Security.'
          }),
          response,
          statusCode: 400
        });
      }
      throw error;
    }
  }));
  router.get({
    path: '/internal/enterprise_search/connectors/{connectorId}/sync_jobs',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        from: _configSchema.schema.number({
          defaultValue: 0,
          min: 0
        }),
        size: _configSchema.schema.number({
          defaultValue: 10,
          min: 0
        }),
        type: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const result = await (0, _searchConnectors.fetchSyncJobs)(client.asCurrentUser, request.params.connectorId, request.query.from, request.query.size, request.query.type);
    return response.ok({
      body: result
    });
  }));
  router.put({
    path: '/internal/enterprise_search/connectors/{connectorId}/pipeline',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        extract_binary_content: _configSchema.schema.boolean(),
        name: _configSchema.schema.string(),
        reduce_whitespace: _configSchema.schema.boolean(),
        run_ml_inference: _configSchema.schema.boolean()
      }),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    await (0, _update_pipeline.updateConnectorPipeline)(client, request.params.connectorId, request.body);
    return response.ok();
  }));
  router.put({
    path: '/internal/enterprise_search/connectors/default_pipeline',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        extract_binary_content: _configSchema.schema.boolean(),
        name: _configSchema.schema.string(),
        reduce_whitespace: _configSchema.schema.boolean(),
        run_ml_inference: _configSchema.schema.boolean()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    await (0, _update_default_pipeline.updateDefaultPipeline)(client, request.body);
    return response.ok();
  }));
  router.get({
    path: '/internal/enterprise_search/connectors/default_pipeline',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {}
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, _, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const result = await (0, _get_default_pipeline.getDefaultPipeline)(client);
    return response.ok({
      body: result
    });
  }));
  router.put({
    path: '/internal/enterprise_search/connectors/{connectorId}/service_type',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        serviceType: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const result = await (0, _searchConnectors.updateConnectorServiceType)(client.asCurrentUser, request.params.connectorId, request.body.serviceType);
    return response.ok({
      body: result
    });
  }));
  router.put({
    path: '/internal/enterprise_search/connectors/{connectorId}/status',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        status: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const result = await (0, _searchConnectors.updateConnectorStatus)(client.asCurrentUser, request.params.connectorId, request.body.status);
    return response.ok({
      body: result
    });
  }));
  router.put({
    path: '/internal/enterprise_search/connectors/{connectorId}/name_and_description',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        description: _configSchema.schema.nullable(_configSchema.schema.string()),
        name: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      name,
      description
    } = request.body;
    const result = await (0, _searchConnectors.updateConnectorNameAndDescription)(client.asCurrentUser, request.params.connectorId, {
      description,
      name
    });
    return response.ok({
      body: result
    });
  }));
  router.put({
    path: '/internal/enterprise_search/connectors/{connectorId}/filtering/draft',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        advanced_snippet: _configSchema.schema.string(),
        filtering_rules: _configSchema.schema.arrayOf(_configSchema.schema.object({
          created_at: _configSchema.schema.string(),
          field: _configSchema.schema.string(),
          id: _configSchema.schema.string(),
          order: _configSchema.schema.number(),
          policy: _configSchema.schema.string(),
          rule: _configSchema.schema.string(),
          updated_at: _configSchema.schema.string(),
          value: _configSchema.schema.string()
        }))
      }),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      connectorId
    } = request.params;
    const {
      advanced_snippet,
      filtering_rules
    } = request.body;
    const result = await (0, _searchConnectors.updateFilteringDraft)(client.asCurrentUser, connectorId, {
      advancedSnippet: advanced_snippet,
      // Have to cast here because our API schema validator doesn't know how to deal with enums
      // We're relying on the schema in the validator above to flag if something goes wrong
      filteringRules: filtering_rules
    });
    return result ? response.ok({
      body: result
    }) : response.conflict();
  }));
  router.put({
    path: '/internal/enterprise_search/connectors/{connectorId}/filtering',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      body: _configSchema.schema.maybe(_configSchema.schema.object({
        advanced_snippet: _configSchema.schema.string(),
        filtering_rules: _configSchema.schema.arrayOf(_configSchema.schema.object({
          created_at: _configSchema.schema.string(),
          field: _configSchema.schema.string(),
          id: _configSchema.schema.string(),
          order: _configSchema.schema.number(),
          policy: _configSchema.schema.string(),
          rule: _configSchema.schema.string(),
          updated_at: _configSchema.schema.string(),
          value: _configSchema.schema.string()
        }))
      })),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      connectorId
    } = request.params;
    const result = await (0, _searchConnectors.updateFiltering)(client.asCurrentUser, connectorId);
    return result ? response.ok({
      body: result
    }) : response.conflict();
  }));
  router.put({
    path: '/internal/enterprise_search/connectors/{connectorId}/native',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        is_native: _configSchema.schema.boolean()
      }),
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const connectorId = decodeURIComponent(request.params.connectorId);
    const {
      is_native
    } = request.body;
    const result = await (0, _searchConnectors.putUpdateNative)(client.asCurrentUser, connectorId, is_native);
    return result ? response.ok({
      body: result
    }) : response.conflict();
  }));
  router.get({
    path: '/internal/enterprise_search/connectors',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      query: _configSchema.schema.object({
        fetchCrawlersOnly: _configSchema.schema.maybe(_configSchema.schema.boolean()),
        from: _configSchema.schema.number({
          defaultValue: 0,
          min: 0
        }),
        searchQuery: _configSchema.schema.string({
          defaultValue: ''
        }),
        size: _configSchema.schema.number({
          defaultValue: 10,
          min: 0
        })
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      fetchCrawlersOnly,
      from,
      size,
      searchQuery
    } = request.query;
    let connectorResult;
    let connectorCountResult;
    let connectorResultSlice;
    const indicesExist = {};
    try {
      connectorResult = await (0, _searchConnectors.fetchConnectors)(client.asCurrentUser, undefined, fetchCrawlersOnly, searchQuery);
      connectorResultSlice = connectorResult.slice(from, from + size);
      for (const connector of connectorResultSlice) {
        if (connector.index_name) {
          const indexExists = await client.asCurrentUser.indices.exists({
            index: connector.index_name
          });
          indicesExist[connector.index_name] = indexExists;
        }
      }
      const indicesSlice = connectorResultSlice.reduce((acc, connector) => {
        if (connector.index_name) {
          acc.push(connector.index_name);
        }
        return acc;
      }, []);
      connectorCountResult = await (0, _fetch_index_counts.fetchIndexCounts)(client, indicesSlice);
    } catch (error) {
      if ((0, _identify_exceptions2.isExpensiveQueriesNotAllowedException)(error)) {
        return (0, _create_error.createError)({
          errorCode: _error_codes.ErrorCode.EXPENSIVE_QUERY_NOT_ALLOWED_ERROR,
          message: _i18n.i18n.translate('xpack.enterpriseSearch.server.routes.connectors.expensive_query_not_allowed_error', {
            defaultMessage: 'Expensive search queries not allowed. "search.allow_expensive_queries" is set to false '
          }),
          response,
          statusCode: 400
        });
      }
      throw error;
    }
    return response.ok({
      body: {
        connectors: connectorResultSlice,
        counts: connectorCountResult,
        indexExists: indicesExist,
        meta: {
          page: {
            from,
            size,
            total: connectorResult.length
          }
        }
      }
    });
  }));
  router.get({
    path: '/internal/enterprise_search/connectors/{connectorId}',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      connectorId
    } = request.params;
    const connectorResult = await (0, _searchConnectors.fetchConnectorById)(client.asCurrentUser, connectorId);
    return response.ok({
      body: {
        connector: connectorResult
      }
    });
  }));
  router.delete({
    path: '/internal/enterprise_search/connectors/{connectorId}',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        shouldDeleteIndex: _configSchema.schema.maybe(_configSchema.schema.boolean())
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      connectorId
    } = request.params;
    const {
      shouldDeleteIndex
    } = request.query;
    let connectorResponse;
    try {
      const connector = await (0, _searchConnectors.fetchConnectorById)(client.asCurrentUser, connectorId);
      const indexNameToDelete = shouldDeleteIndex ? connector === null || connector === void 0 ? void 0 : connector.index_name : null;
      const apiKeyId = connector === null || connector === void 0 ? void 0 : connector.api_key_id;
      const secretId = connector === null || connector === void 0 ? void 0 : connector.api_key_secret_id;
      connectorResponse = await (0, _searchConnectors.deleteConnectorById)(client.asCurrentUser, connectorId);
      if (indexNameToDelete) {
        await (0, _delete_pipelines.deleteIndexPipelines)(client, indexNameToDelete);
        await (0, _delete_access_control_index.deleteAccessControlIndex)(client, indexNameToDelete);
        const indexExists = await client.asCurrentUser.indices.exists({
          index: indexNameToDelete
        });
        if (indexExists) {
          await client.asCurrentUser.indices.delete({
            index: indexNameToDelete
          });
        }
      }
      if (apiKeyId) {
        await client.asCurrentUser.security.invalidateApiKey({
          ids: [apiKeyId]
        });
      }
      if (secretId) {
        await (0, _searchConnectors.deleteConnectorSecret)(client.asCurrentUser, secretId);
      }
    } catch (error) {
      if ((0, _identify_exceptions.isResourceNotFoundException)(error)) {
        return (0, _create_error.createError)({
          errorCode: _error_codes.ErrorCode.RESOURCE_NOT_FOUND,
          message: _i18n.i18n.translate('xpack.enterpriseSearch.server.routes.connectors.resource_not_found_error', {
            defaultMessage: 'Connector with id {connectorId} is not found.',
            values: {
              connectorId
            }
          }),
          response,
          statusCode: 404
        });
      }
      if ((0, _identify_exceptions2.isIndexNotFoundException)(error)) {
        return (0, _create_error.createError)({
          errorCode: _error_codes.ErrorCode.INDEX_NOT_FOUND,
          message: 'Could not find index',
          response,
          statusCode: 404
        });
      }
      throw error;
    }
    return response.ok({
      body: connectorResponse
    });
  }));
  router.put({
    path: '/internal/enterprise_search/connectors/{connectorId}/index_name/{indexName}',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string(),
        indexName: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      connectorId,
      indexName
    } = request.params;
    try {
      await client.asCurrentUser.transport.request({
        body: {
          index_name: indexName
        },
        method: 'PUT',
        path: `/_connector/${connectorId}/_index_name`
      });
    } catch (error) {
      var _error$meta$body, _error$meta$body$erro;
      // This will almost always be a conflict because another connector has the index configured
      // ES returns this reason nicely so we can just pass it through
      return response.customError({
        body: (_error$meta$body = error.meta.body) === null || _error$meta$body === void 0 ? void 0 : (_error$meta$body$erro = _error$meta$body.error) === null || _error$meta$body$erro === void 0 ? void 0 : _error$meta$body$erro.reason,
        statusCode: 500
      });
    }
    const connector = await (0, _searchConnectors.fetchConnectorById)(client.asCurrentUser, connectorId);
    if (connector !== null && connector !== void 0 && connector.is_native) {
      // generateApiKey will search for the connector doc based on index_name, so we need to refresh the index before that.
      await client.asCurrentUser.indices.refresh({
        index: _searchConnectors.CONNECTORS_INDEX
      });
      await (0, _generate_api_key.generateApiKey)(client, indexName, true);
    }
    return response.ok();
  }));
  router.get({
    path: '/internal/enterprise_search/connectors/available_indices',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      query: _configSchema.schema.object({
        from: _configSchema.schema.number({
          defaultValue: 0,
          min: 0
        }),
        search_query: _configSchema.schema.maybe(_configSchema.schema.string()),
        size: _configSchema.schema.number({
          defaultValue: 40,
          min: 0
        })
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      from,
      size,
      search_query: searchQuery
    } = request.query;
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      indexNames,
      totalResults
    } = await (0, _fetch_unattached_indices.fetchUnattachedIndices)(client, searchQuery, from, size);
    return response.ok({
      body: {
        indexNames,
        meta: {
          page: {
            from,
            size,
            total: totalResults
          }
        }
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.post({
    path: '/internal/enterprise_search/connectors/{connectorId}/generate_config',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      params: _configSchema.schema.object({
        connectorId: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      connectorId
    } = request.params;
    let associatedIndex;
    let apiKeyResponse;
    try {
      const connector = await (0, _searchConnectors.fetchConnectorById)(client.asCurrentUser, connectorId);
      if (!connector) {
        return (0, _create_error.createError)({
          errorCode: _error_codes.ErrorCode.RESOURCE_NOT_FOUND,
          message: _i18n.i18n.translate('xpack.enterpriseSearch.server.routes.connectors.resource_not_found_error', {
            defaultMessage: 'Connector with id {connectorId} is not found.',
            values: {
              connectorId
            }
          }),
          response,
          statusCode: 404
        });
      }
      const configResponse = await (0, _generate_config.generateConfig)(client, connector);
      associatedIndex = configResponse.associatedIndex;
      apiKeyResponse = configResponse.apiKeyResponse;
    } catch (error) {
      if (error.message === _error_codes.ErrorCode.GENERATE_INDEX_NAME_ERROR) {
        (0, _create_error.createError)({
          errorCode: _error_codes.ErrorCode.GENERATE_INDEX_NAME_ERROR,
          message: _i18n.i18n.translate('xpack.enterpriseSearch.server.routes.connectors.generateConfiguration.indexAlreadyExistsError', {
            defaultMessage: 'Cannot find a unique index name to generate configuration'
          }),
          response,
          statusCode: 409
        });
        throw error;
      }
    }
    return response.ok({
      body: {
        apiKey: apiKeyResponse,
        connectorId,
        indexName: associatedIndex
      },
      headers: {
        'content-type': 'application/json'
      }
    });
  }));
  router.post({
    path: '/internal/enterprise_search/connectors/generate_connector_name',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the scoped ES client'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        connectorName: _configSchema.schema.maybe(_configSchema.schema.string()),
        connectorType: _configSchema.schema.string()
      })
    }
  }, (0, _elasticsearch_error_handler.elasticsearchErrorHandler)(log, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      connectorType,
      connectorName
    } = request.body;
    try {
      const generatedNames = await (0, _generate_connector_name.generateConnectorName)(client, connectorType !== null && connectorType !== void 0 ? connectorType : 'custom', connectorName);
      return response.ok({
        body: generatedNames,
        headers: {
          'content-type': 'application/json'
        }
      });
    } catch (error) {
      if (error.message === _error_codes.ErrorCode.GENERATE_INDEX_NAME_ERROR) {
        return (0, _create_error.createError)({
          errorCode: _error_codes.ErrorCode.GENERATE_INDEX_NAME_ERROR,
          message: _i18n.i18n.translate('xpack.enterpriseSearch.server.routes.connectors.generateConfiguration.indexAlreadyExistsError', {
            defaultMessage: 'Cannot find a unique connector name'
          }),
          response,
          statusCode: 409
        });
      } else {
        throw error;
      }
    }
  }));
}