"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GenericEndpointInlineEditableTableLogic = void 0;
var _kea = require("kea");
var _flash_messages = require("../../flash_messages");
var _handle_api_errors = require("../../flash_messages/handle_api_errors");
var _http = require("../../http");
var _inline_editable_table_logic = require("../inline_editable_table/inline_editable_table_logic");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const GenericEndpointInlineEditableTableLogic = exports.GenericEndpointInlineEditableTableLogic = (0, _kea.kea)({
  path: key => ['enterprise_search', 'generic_endpoint_inline_editable_table_logic', key],
  key: props => props.instanceId,
  actions: () => ({
    addItem: (item, onSuccess) => ({
      item,
      onSuccess
    }),
    setLoading: true,
    clearLoading: true,
    deleteItem: (item, onSuccess) => ({
      item,
      onSuccess
    }),
    reorderItems: (items, oldItems, onSuccess) => ({
      items,
      oldItems,
      onSuccess
    }),
    updateItem: (item, onSuccess) => ({
      item,
      onSuccess
    })
  }),
  reducers: () => ({
    isLoading: [false, {
      addItem: () => true,
      setLoading: () => true,
      clearLoading: () => false,
      deleteItem: () => true,
      updateItem: () => true
    }]
  }),
  listeners: ({
    actions,
    props
  }) => ({
    addItem: async ({
      item,
      onSuccess
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        addRoute,
        onAdd,
        dataProperty
      } = props;
      try {
        const response = await http.post(addRoute, {
          body: JSON.stringify(item)
        });
        const itemsFromResponse = response[dataProperty];
        onAdd(item, itemsFromResponse);
        onSuccess();
      } catch (e) {
        const errors = (0, _handle_api_errors.getErrorsFromHttpResponse)(e);
        (0, _inline_editable_table_logic.InlineEditableTableLogic)({
          instanceId: props.instanceId
        }).actions.setRowErrors(errors);
      } finally {
        actions.clearLoading();
      }
    },
    deleteItem: async ({
      item,
      onSuccess
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        deleteRoute,
        onDelete,
        dataProperty
      } = props;
      try {
        const response = await http.delete(deleteRoute(item));
        const itemsFromResponse = response[dataProperty];
        onDelete(item, itemsFromResponse);
        onSuccess();
      } catch (e) {
        const errors = (0, _handle_api_errors.getErrorsFromHttpResponse)(e);
        (0, _inline_editable_table_logic.InlineEditableTableLogic)({
          instanceId: props.instanceId
        }).actions.setRowErrors(errors);
      } finally {
        actions.clearLoading();
      }
    },
    updateItem: async ({
      item,
      onSuccess
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        updateRoute,
        onUpdate,
        dataProperty
      } = props;
      const dataToSubmit = (0, _utils.stripIdAndCreatedAtFromItem)(item);
      try {
        const response = await http.put(updateRoute(item), {
          body: JSON.stringify(dataToSubmit)
        });
        const itemsFromResponse = response[dataProperty];
        onUpdate(item, itemsFromResponse);
        onSuccess();
      } catch (e) {
        const errors = (0, _handle_api_errors.getErrorsFromHttpResponse)(e);
        (0, _inline_editable_table_logic.InlineEditableTableLogic)({
          instanceId: props.instanceId
        }).actions.setRowErrors(errors);
      } finally {
        actions.clearLoading();
      }
    },
    reorderItems: async ({
      items,
      oldItems,
      onSuccess
    }) => {
      const {
        reorderRoute,
        onReorder,
        dataProperty
      } = props;
      if (!reorderRoute || !onReorder) return;
      const {
        http
      } = _http.HttpLogic.values;
      const reorderedItemIds = items.map(({
        id
      }, itemIndex) => ({
        id,
        order: itemIndex
      }));
      onReorder(items); // We optimistically reorder this so that the client-side UI doesn't snap back while waiting for the http response

      try {
        actions.setLoading();
        const response = await http.put(reorderRoute, {
          body: JSON.stringify({
            [dataProperty]: reorderedItemIds
          })
        });
        const itemsFromResponse = response[dataProperty];
        onReorder(itemsFromResponse); // Final reorder based on server response
        onSuccess();
      } catch (e) {
        onReorder(oldItems);
        (0, _flash_messages.toastAPIErrors)(e);
      }
      actions.clearLoading();
    }
  })
});