"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CurationsSettingsLogic = void 0;
var _kea = require("kea");
var _flash_messages = require("../../../../../shared/flash_messages");
var _http = require("../../../../../shared/http");
var _engine = require("../../../engine");
var _curations_logic = require("../../curations_logic");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CurationsSettingsLogic = exports.CurationsSettingsLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'curations', 'curations_settings_logic'],
  actions: () => ({
    loadCurationsSettings: true,
    onCurationsSettingsLoad: curationsSettings => ({
      curationsSettings
    }),
    onSkipLoadingCurationsSettings: true,
    toggleCurationsEnabled: true,
    toggleCurationsMode: true,
    updateCurationsSetting: currationsSetting => ({
      currationsSetting
    })
  }),
  reducers: () => ({
    dataLoading: [true, {
      onCurationsSettingsLoad: () => false,
      onSkipLoadingCurationsSettings: () => false
    }],
    curationsSettings: [{
      enabled: false,
      mode: 'manual'
    }, {
      onCurationsSettingsLoad: (_, {
        curationsSettings
      }) => curationsSettings
    }]
  }),
  listeners: ({
    actions,
    values
  }) => ({
    loadCurationsSettings: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      try {
        const response = await http.get(`/internal/app_search/engines/${engineName}/adaptive_relevance/settings`);
        actions.onCurationsSettingsLoad(response.curation);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    toggleCurationsEnabled: async () => {
      if (values.curationsSettings.enabled) {
        actions.updateCurationsSetting({
          enabled: false,
          mode: 'manual'
        });
      } else {
        actions.updateCurationsSetting({
          enabled: true
        });
      }
    },
    toggleCurationsMode: async () => {
      actions.updateCurationsSetting({
        mode: values.curationsSettings.mode === 'automatic' ? 'manual' : 'automatic'
      });
    },
    updateCurationsSetting: async ({
      currationsSetting
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      try {
        const response = await http.put(`/internal/app_search/engines/${engineName}/adaptive_relevance/settings`, {
          body: JSON.stringify({
            curation: currationsSetting
          })
        });
        actions.onCurationsSettingsLoad(response.curation);

        //  Re-fetch data so that UI updates to new settings
        _curations_logic.CurationsLogic.actions.loadCurations();
        _engine.EngineLogic.actions.initializeEngine();
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    }
  })
});