"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateAttackDiscoveryStatusToRunning = exports.updateAttackDiscoveryStatusToCanceled = exports.updateAttackDiscoveryLastViewedAt = exports.updateAttackDiscoveries = exports.getAttackDiscoveryStats = exports.attackDiscoveryStatus = exports.addGenerationInterval = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _moment = _interopRequireDefault(require("moment/moment"));
var _telemetry = require("./telemetry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const attackDiscoveryStatus = exports.attackDiscoveryStatus = {
  canceled: 'canceled',
  failed: 'failed',
  running: 'running',
  succeeded: 'succeeded'
};
const MAX_GENERATION_INTERVALS = 5;
const addGenerationInterval = (generationIntervals, generationInterval) => {
  const newGenerationIntervals = [generationInterval, ...generationIntervals];
  if (newGenerationIntervals.length > MAX_GENERATION_INTERVALS) {
    return newGenerationIntervals.slice(0, MAX_GENERATION_INTERVALS); // Return the first MAX_GENERATION_INTERVALS items
  }
  return newGenerationIntervals;
};
exports.addGenerationInterval = addGenerationInterval;
const updateAttackDiscoveryStatusToRunning = async (dataClient, authenticatedUser, apiConfig, alertsContextCount) => {
  const foundAttackDiscovery = await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.findAttackDiscoveryByConnectorId({
    connectorId: apiConfig.connectorId,
    authenticatedUser
  }));
  const currentAd = foundAttackDiscovery ? await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.updateAttackDiscovery({
    attackDiscoveryUpdateProps: {
      alertsContextCount,
      backingIndex: foundAttackDiscovery.backingIndex,
      id: foundAttackDiscovery.id,
      status: attackDiscoveryStatus.running
    },
    authenticatedUser
  })) : await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.createAttackDiscovery({
    attackDiscoveryCreate: {
      alertsContextCount,
      apiConfig,
      attackDiscoveries: [],
      status: attackDiscoveryStatus.running
    },
    authenticatedUser
  }));
  if (!currentAd) {
    throw new Error(`Could not ${foundAttackDiscovery ? 'update' : 'create'} attack discovery for connectorId: ${apiConfig.connectorId}`);
  }
  return {
    attackDiscoveryId: currentAd.id,
    currentAd
  };
};
exports.updateAttackDiscoveryStatusToRunning = updateAttackDiscoveryStatusToRunning;
const updateAttackDiscoveryStatusToCanceled = async (dataClient, authenticatedUser, connectorId) => {
  const foundAttackDiscovery = await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.findAttackDiscoveryByConnectorId({
    connectorId,
    authenticatedUser
  }));
  if (foundAttackDiscovery == null) {
    throw new Error(`Could not find attack discovery for connector id: ${connectorId}`);
  }
  if (foundAttackDiscovery.status !== 'running') {
    throw new Error(`Connector id ${connectorId} does not have a running attack discovery, and therefore cannot be canceled.`);
  }
  const updatedAttackDiscovery = await (dataClient === null || dataClient === void 0 ? void 0 : dataClient.updateAttackDiscovery({
    attackDiscoveryUpdateProps: {
      backingIndex: foundAttackDiscovery.backingIndex,
      id: foundAttackDiscovery.id,
      status: attackDiscoveryStatus.canceled
    },
    authenticatedUser
  }));
  if (!updatedAttackDiscovery) {
    throw new Error(`Could not update attack discovery for connector id: ${connectorId}`);
  }
  return updatedAttackDiscovery;
};
exports.updateAttackDiscoveryStatusToCanceled = updateAttackDiscoveryStatusToCanceled;
const updateAttackDiscoveries = async ({
  anonymizedAlerts,
  apiConfig,
  attackDiscoveries,
  executionUuid,
  authenticatedUser,
  dataClient,
  hasFilter,
  end,
  latestReplacements,
  logger,
  size,
  start,
  startTime,
  telemetry
}) => {
  try {
    const currentAd = await dataClient.getAttackDiscovery({
      id: executionUuid,
      authenticatedUser
    });
    if (currentAd === null || (currentAd === null || currentAd === void 0 ? void 0 : currentAd.status) === 'canceled') {
      return;
    }
    const endTime = (0, _moment.default)();
    const durationMs = endTime.diff(startTime);
    const alertsContextCount = anonymizedAlerts.length;
    const updateProps = {
      alertsContextCount,
      attackDiscoveries: attackDiscoveries !== null && attackDiscoveries !== void 0 ? attackDiscoveries : undefined,
      status: attackDiscoveryStatus.succeeded,
      ...(alertsContextCount === 0 ? {} : {
        generationIntervals: addGenerationInterval(currentAd.generationIntervals, {
          durationMs,
          date: new Date().toISOString()
        })
      }),
      id: executionUuid,
      replacements: latestReplacements,
      backingIndex: currentAd.backingIndex
    };
    await dataClient.updateAttackDiscovery({
      attackDiscoveryUpdateProps: updateProps,
      authenticatedUser
    });
    (0, _telemetry.reportAttackDiscoveryGenerationSuccess)({
      alertsContextCount,
      apiConfig,
      attackDiscoveries,
      durationMs,
      end,
      hasFilter,
      size,
      start,
      telemetry
    });
  } catch (updateErr) {
    logger.error(updateErr);
    const updateError = (0, _securitysolutionEsUtils.transformError)(updateErr);
    (0, _telemetry.reportAttackDiscoveryGenerationFailure)({
      apiConfig,
      errorMessage: updateError.message,
      telemetry
    });
  }
};
exports.updateAttackDiscoveries = updateAttackDiscoveries;
const updateAttackDiscoveryLastViewedAt = async ({
  connectorId,
  authenticatedUser,
  dataClient
}) => {
  const attackDiscovery = await dataClient.findAttackDiscoveryByConnectorId({
    connectorId,
    authenticatedUser
  });
  if (attackDiscovery == null) {
    return null;
  }

  // update lastViewedAt time as this is the function used for polling by connectorId
  return dataClient.updateAttackDiscovery({
    attackDiscoveryUpdateProps: {
      id: attackDiscovery.id,
      lastViewedAt: new Date().toISOString(),
      backingIndex: attackDiscovery.backingIndex
    },
    authenticatedUser
  });
};
exports.updateAttackDiscoveryLastViewedAt = updateAttackDiscoveryLastViewedAt;
const getAttackDiscoveryStats = async ({
  authenticatedUser,
  dataClient
}) => {
  const attackDiscoveries = await dataClient.findAllAttackDiscoveries({
    authenticatedUser
  });
  return attackDiscoveries.map(ad => {
    const updatedAt = (0, _moment.default)(ad.updatedAt);
    const lastViewedAt = (0, _moment.default)(ad.lastViewedAt);
    const timeSinceLastViewed = updatedAt.diff(lastViewedAt);
    const hasViewed = timeSinceLastViewed <= 0;
    const discoveryCount = ad.attackDiscoveries.length;
    return {
      hasViewed,
      status: ad.status,
      count: discoveryCount,
      connectorId: ad.apiConfig.connectorId
    };
  });
};
exports.getAttackDiscoveryStats = getAttackDiscoveryStats;