"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AttackDiscoveryDataClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _ai_assistant_data_clients = require("../../../ai_assistant_data_clients");
var _find = require("../../../ai_assistant_data_clients/find");
var _find_all_attack_discoveries = require("./find_all_attack_discoveries/find_all_attack_discoveries");
var _combine_find_attack_discovery_filters = require("./combine_find_attack_discovery_filters");
var _find_attack_discovery_by_connector_id = require("./find_attack_discovery_by_connector_id/find_attack_discovery_by_connector_id");
var _update_attack_discovery = require("./update_attack_discovery/update_attack_discovery");
var _create_attack_discovery = require("./create_attack_discovery/create_attack_discovery");
var _create_attack_discovery_alerts = require("./create_attack_discovery_alerts");
var _get_attack_discovery = require("./get_attack_discovery/get_attack_discovery");
var _get_attack_discovery_generations = require("./get_attack_discovery_generations");
var _get_attack_discovery_generation_by_id_query = require("./get_attack_discovery_generation_by_id_query");
var _get_attack_discovery_generations_query = require("./get_attack_discovery_generations_query");
var _get_combined_filter = require("./get_combined_filter");
var _get_find_attack_discovery_alerts_aggregation = require("./get_find_attack_discovery_alerts_aggregation");
var _transform_search_response_to_alerts = require("./transforms/transform_search_response_to_alerts");
var _get_scheduled_index_pattern = require("./get_scheduled_index_pattern");
var _get_update_attack_discovery_alerts_query = require("../get_update_attack_discovery_alerts_query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FIRST_PAGE = 1; // CAUTION: sever-side API uses a 1-based page index convention (for consistency with similar existing APIs)
const DEFAULT_PER_PAGE = 10;
class AttackDiscoveryDataClient extends _ai_assistant_data_clients.AIAssistantDataClient {
  constructor(options) {
    super(options);
    (0, _defineProperty2.default)(this, "adhocAttackDiscoveryDataClient", void 0);
    /**
     * Fetches an attack discovery
     * @param options
     * @param options.id The existing attack discovery id.
     * @param options.authenticatedUser Current authenticated user.
     * @returns The attack discovery response
     */
    (0, _defineProperty2.default)(this, "getAttackDiscovery", async ({
      id,
      authenticatedUser
    }) => {
      const esClient = await this.options.elasticsearchClientPromise;
      return (0, _get_attack_discovery.getAttackDiscovery)({
        esClient,
        logger: this.options.logger,
        attackDiscoveryIndex: this.indexTemplateAndPattern.alias,
        id,
        user: authenticatedUser
      });
    });
    /**
     * Creates an attack discovery, if given at least the "apiConfig"
     * @param options
     * @param options.attackDiscoveryCreate
     * @param options.authenticatedUser
     * @returns The Attack Discovery created
     */
    (0, _defineProperty2.default)(this, "createAttackDiscovery", async ({
      attackDiscoveryCreate,
      authenticatedUser
    }) => {
      const esClient = await this.options.elasticsearchClientPromise;
      return (0, _create_attack_discovery.createAttackDiscovery)({
        esClient,
        logger: this.options.logger,
        attackDiscoveryIndex: this.indexTemplateAndPattern.alias,
        spaceId: this.spaceId,
        user: authenticatedUser,
        attackDiscoveryCreate
      });
    });
    (0, _defineProperty2.default)(this, "getAdHocAlertsIndexPattern", () => {
      if (this.adhocAttackDiscoveryDataClient === undefined) {
        throw new Error('`adhocAttackDiscoveryDataClient` is required');
      }
      return this.adhocAttackDiscoveryDataClient.indexNameWithNamespace(this.spaceId);
    });
    (0, _defineProperty2.default)(this, "getScheduledAndAdHocIndexPattern", () => {
      return [(0, _get_scheduled_index_pattern.getScheduledIndexPattern)(this.spaceId),
      // scheduled
      this.getAdHocAlertsIndexPattern() // ad-hoc
      ].join(',');
    });
    (0, _defineProperty2.default)(this, "createAttackDiscoveryAlerts", async ({
      authenticatedUser,
      createAttackDiscoveryAlertsParams
    }) => {
      if (this.adhocAttackDiscoveryDataClient === undefined) {
        throw new Error('`adhocAttackDiscoveryDataClient` is required');
      }
      return (0, _create_attack_discovery_alerts.createAttackDiscoveryAlerts)({
        adhocAttackDiscoveryDataClient: this.adhocAttackDiscoveryDataClient,
        authenticatedUser,
        createAttackDiscoveryAlertsParams,
        logger: this.options.logger,
        spaceId: this.spaceId
      });
    });
    // Runs an aggregation only bound to the (optional) alertIds and date range
    // to prevent the connector names from being filtered-out as the user applies more filters:
    (0, _defineProperty2.default)(this, "getAlertConnectorNames", async ({
      alertIds,
      authenticatedUser,
      end,
      esClient,
      ids,
      index,
      logger,
      page,
      perPage,
      sortField,
      sortOrder,
      start
    }) => {
      const aggs = (0, _get_find_attack_discovery_alerts_aggregation.getFindAttackDiscoveryAlertsAggregation)();

      // just use the (optional) alertIds and date range to prevent the connector
      // names from being filtered-out as the user applies more filters:
      const connectorsAggsFilter = (0, _combine_find_attack_discovery_filters.combineFindAttackDiscoveryFilters)({
        alertIds,
        // optional
        end,
        ids,
        start
      });
      const combinedConnectorsAggsFilter = (0, _get_combined_filter.getCombinedFilter)({
        authenticatedUser,
        filter: connectorsAggsFilter
      });
      const aggsResult = await (0, _find.findDocuments)({
        aggs,
        esClient,
        filter: combinedConnectorsAggsFilter,
        index,
        logger,
        page,
        perPage,
        sortField,
        sortOrder: sortOrder
      });
      const {
        connectorNames
      } = (0, _transform_search_response_to_alerts.transformSearchResponseToAlerts)({
        logger,
        response: aggsResult.data
      });
      return connectorNames;
    });
    (0, _defineProperty2.default)(this, "findAttackDiscoveryAlerts", async ({
      authenticatedUser,
      esClient,
      findAttackDiscoveryAlertsParams,
      logger
    }) => {
      const aggs = (0, _get_find_attack_discovery_alerts_aggregation.getFindAttackDiscoveryAlertsAggregation)();
      const {
        alertIds,
        connectorNames,
        // <-- as a filter input
        end,
        ids,
        search,
        shared,
        sortField = '@timestamp',
        sortOrder = 'desc',
        start,
        status,
        page = FIRST_PAGE,
        perPage = DEFAULT_PER_PAGE
      } = findAttackDiscoveryAlertsParams;
      const index = this.getScheduledAndAdHocIndexPattern();
      const filter = (0, _combine_find_attack_discovery_filters.combineFindAttackDiscoveryFilters)({
        alertIds,
        connectorNames,
        end,
        ids,
        search,
        start,
        status
      });
      const combinedFilter = (0, _get_combined_filter.getCombinedFilter)({
        authenticatedUser,
        filter,
        shared
      });
      const result = await (0, _find.findDocuments)({
        aggs,
        esClient,
        filter: combinedFilter,
        index,
        logger,
        page,
        perPage,
        sortField,
        sortOrder: sortOrder
      });
      const {
        data,
        uniqueAlertIdsCount
      } = (0, _transform_search_response_to_alerts.transformSearchResponseToAlerts)({
        logger,
        response: result.data
      });
      const alertConnectorNames = await this.getAlertConnectorNames({
        alertIds,
        authenticatedUser,
        end,
        esClient,
        ids,
        index,
        logger,
        page,
        perPage,
        sortField,
        sortOrder,
        start
      });
      return {
        connector_names: alertConnectorNames,
        // <-- from the separate aggregation
        data,
        page: result.page,
        per_page: result.perPage,
        total: result.total,
        unique_alert_ids_count: uniqueAlertIdsCount
      };
    });
    (0, _defineProperty2.default)(this, "getAttackDiscoveryGenerations", async ({
      authenticatedUser,
      eventLogIndex,
      getAttackDiscoveryGenerationsParams,
      logger,
      spaceId
    }) => {
      const esClient = await this.options.elasticsearchClientPromise;
      const {
        size,
        start,
        end
      } = getAttackDiscoveryGenerationsParams;
      const generationsQuery = (0, _get_attack_discovery_generations_query.getAttackDiscoveryGenerationsQuery)({
        authenticatedUser,
        end,
        eventLogIndex,
        size,
        spaceId,
        start
      });
      return (0, _get_attack_discovery_generations.getAttackDiscoveryGenerations)({
        authenticatedUser,
        esClient,
        eventLogIndex,
        generationsQuery,
        getAttackDiscoveryGenerationsParams,
        logger,
        spaceId
      });
    });
    (0, _defineProperty2.default)(this, "bulkUpdateAttackDiscoveryAlerts", async ({
      authenticatedUser,
      esClient,
      ids,
      kibanaAlertWorkflowStatus,
      logger,
      visibility
    }) => {
      const PER_PAGE = 1000;
      const indexPattern = this.getScheduledAndAdHocIndexPattern();
      if (ids.length === 0) {
        logger.debug(() => `No Attack discovery alerts to update for index ${indexPattern} in bulkUpdateAttackDiscoveryAlerts`);
        return [];
      }
      try {
        logger.debug(() => `Updating Attack discovery alerts in index ${indexPattern} with alert ids: ${ids.join(', ')}`);
        const updateByQuery = (0, _get_update_attack_discovery_alerts_query.getUpdateAttackDiscoveryAlertsQuery)({
          authenticatedUser,
          ids,
          indexPattern,
          kibanaAlertWorkflowStatus,
          visibility
        });
        const updateResponse = await esClient.updateByQuery(updateByQuery);
        await esClient.indices.refresh({
          allow_no_indices: true,
          ignore_unavailable: true,
          index: indexPattern
        });
        if (updateResponse.failures != null && updateResponse.failures.length > 0) {
          const errorDetails = updateResponse.failures.flatMap(failure => {
            const error = failure === null || failure === void 0 ? void 0 : failure.cause;
            if (error == null) {
              return [];
            }
            const id = failure.id != null ? ` id: ${failure.id}` : '';
            const details = `\nError updating attack discovery alert${id} ${error}`;
            return [details];
          });
          const allErrorDetails = errorDetails.join(', ');
          throw new Error(`Failed to update attack discovery alerts ${allErrorDetails}`);
        }
        const alertsResult = await this.findAttackDiscoveryAlerts({
          authenticatedUser,
          esClient,
          findAttackDiscoveryAlertsParams: {
            ids,
            page: FIRST_PAGE,
            perPage: PER_PAGE,
            sortField: '@timestamp'
          },
          logger
        });
        return alertsResult.data;
      } catch (err) {
        logger.error(`Error updating Attack discovery alerts: ${err} for ids: ${ids.join(', ')}`);
        throw err;
      }
    });
    (0, _defineProperty2.default)(this, "getAttackDiscoveryGenerationById", async ({
      authenticatedUser,
      eventLogIndex,
      executionUuid,
      logger,
      spaceId
    }) => {
      const esClient = await this.options.elasticsearchClientPromise;
      const generationByIdQuery = (0, _get_attack_discovery_generation_by_id_query.getAttackDiscoveryGenerationByIdQuery)({
        authenticatedUser,
        eventLogIndex,
        executionUuid,
        spaceId
      });
      const result = await (0, _get_attack_discovery_generations.getAttackDiscoveryGenerations)({
        authenticatedUser,
        esClient,
        eventLogIndex,
        generationsQuery: generationByIdQuery,
        getAttackDiscoveryGenerationsParams: {
          size: 1
        },
        logger,
        spaceId
      });
      if ((result === null || result === void 0 ? void 0 : result.generations[0]) == null) {
        throw new Error(`Generation with execution_uuid ${executionUuid} not found`);
      }
      return result === null || result === void 0 ? void 0 : result.generations[0];
    });
    /**
     * Find attack discovery by apiConfig connectorId
     * @param options
     * @param options.connectorId
     * @param options.authenticatedUser
     * @returns The Attack Discovery found
     */
    (0, _defineProperty2.default)(this, "findAttackDiscoveryByConnectorId", async ({
      connectorId,
      authenticatedUser
    }) => {
      const esClient = await this.options.elasticsearchClientPromise;
      return (0, _find_attack_discovery_by_connector_id.findAttackDiscoveryByConnectorId)({
        esClient,
        logger: this.options.logger,
        attackDiscoveryIndex: this.indexTemplateAndPattern.alias,
        connectorId,
        user: authenticatedUser
      });
    });
    /**
     * Finds all attack discovery for authenticated user
     * @param options
     * @param options.authenticatedUser
     * @returns The Attack Discovery
     */
    (0, _defineProperty2.default)(this, "findAllAttackDiscoveries", async ({
      authenticatedUser
    }) => {
      const esClient = await this.options.elasticsearchClientPromise;
      return (0, _find_all_attack_discoveries.findAllAttackDiscoveries)({
        esClient,
        logger: this.options.logger,
        attackDiscoveryIndex: this.indexTemplateAndPattern.alias,
        user: authenticatedUser
      });
    });
    /**
     * Updates an attack discovery
     * @param options
     * @param options.attackDiscoveryUpdateProps
     * @param options.authenticatedUser
     */
    (0, _defineProperty2.default)(this, "updateAttackDiscovery", async ({
      attackDiscoveryUpdateProps,
      authenticatedUser
    }) => {
      const esClient = await this.options.elasticsearchClientPromise;
      return (0, _update_attack_discovery.updateAttackDiscovery)({
        esClient,
        logger: this.options.logger,
        attackDiscoveryIndex: attackDiscoveryUpdateProps.backingIndex,
        attackDiscoveryUpdateProps,
        user: authenticatedUser
      });
    });
    this.options = options;
    this.adhocAttackDiscoveryDataClient = this.options.adhocAttackDiscoveryDataClient;
  }
  async refreshEventLogIndex(eventLogIndex) {
    const esClient = await this.options.elasticsearchClientPromise;
    await esClient.indices.refresh({
      allow_no_indices: true,
      ignore_unavailable: true,
      index: eventLogIndex
    });
  }
}
exports.AttackDiscoveryDataClient = AttackDiscoveryDataClient;