"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateConversation = exports.transformToUpdateScheme = void 0;
var _get_conversation = require("./get_conversation");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const updateConversation = async ({
  esClient,
  logger,
  conversationIndex,
  conversationUpdateProps,
  isPatch,
  user
}) => {
  var _response8, _response9;
  const updatedAt = new Date().toISOString();
  const params = transformToUpdateScheme(updatedAt, conversationUpdateProps);
  const maxRetries = 3;
  let attempt = 0;
  let response;
  while (attempt < maxRetries) {
    try {
      var _response, _response2, _response3, _response4, _response5, _response6, _response7;
      response = await esClient.updateByQuery({
        conflicts: 'proceed',
        index: conversationIndex,
        query: {
          ids: {
            values: [params.id]
          }
        },
        refresh: true,
        script: (0, _helpers.getUpdateScript)({
          conversation: params,
          isPatch
        }).script
      });
      if ((_response = response) !== null && _response !== void 0 && _response.updated && ((_response2 = response) === null || _response2 === void 0 ? void 0 : _response2.updated) > 0 || (_response3 = response) !== null && _response3 !== void 0 && _response3.failures && ((_response4 = response) === null || _response4 === void 0 ? void 0 : _response4.failures.length) > 0) {
        break;
      }
      if ((_response5 = response) !== null && _response5 !== void 0 && _response5.version_conflicts && ((_response6 = response) === null || _response6 === void 0 ? void 0 : _response6.version_conflicts) > 0 && ((_response7 = response) === null || _response7 === void 0 ? void 0 : _response7.updated) === 0) {
        attempt++;
        if (attempt < maxRetries) {
          logger.warn(`Version conflict detected, retrying updateConversation (attempt ${attempt + 1}) for conversation ID: ${params.id}`);
          await new Promise(resolve => setTimeout(resolve, 100 * attempt));
        }
      } else {
        break;
      }
    } catch (err) {
      logger.warn(`Error updating conversation: ${err} by ID: ${params.id}`);
      throw err;
    }
  }
  if (response && (_response8 = response) !== null && _response8 !== void 0 && _response8.failures && ((_response9 = response) === null || _response9 === void 0 ? void 0 : _response9.failures.length) > 0) {
    var _response10;
    logger.warn(`Error updating conversation: ${(_response10 = response) === null || _response10 === void 0 ? void 0 : _response10.failures.map(f => f.id)} by ID: ${params.id}`);
    return null;
  }
  const updatedConversation = await (0, _get_conversation.getConversation)({
    esClient,
    conversationIndex,
    id: params.id,
    logger,
    user
  });
  return updatedConversation;
};
exports.updateConversation = updateConversation;
const transformToUpdateScheme = (updatedAt, {
  title,
  apiConfig,
  excludeFromLastConversationStorage,
  messages,
  replacements,
  id
}) => {
  return {
    id,
    updated_at: updatedAt,
    ...(title ? {
      title
    } : {}),
    ...(apiConfig ? {
      api_config: {
        action_type_id: apiConfig === null || apiConfig === void 0 ? void 0 : apiConfig.actionTypeId,
        connector_id: apiConfig === null || apiConfig === void 0 ? void 0 : apiConfig.connectorId,
        default_system_prompt_id: apiConfig === null || apiConfig === void 0 ? void 0 : apiConfig.defaultSystemPromptId,
        model: apiConfig === null || apiConfig === void 0 ? void 0 : apiConfig.model,
        provider: apiConfig === null || apiConfig === void 0 ? void 0 : apiConfig.provider
      }
    } : {}),
    ...(excludeFromLastConversationStorage != null ? {
      exclude_from_last_conversation_storage: excludeFromLastConversationStorage
    } : {}),
    ...(replacements ? {
      replacements: Object.keys(replacements).map(key => ({
        uuid: key,
        value: replacements[key]
      }))
    } : {}),
    ...(messages ? {
      messages: messages.map(message => ({
        '@timestamp': message.timestamp,
        content: message.content,
        is_error: message.isError,
        reader: message.reader,
        role: message.role,
        ...(message.metadata ? {
          metadata: {
            ...(message.metadata.contentReferences ? {
              content_references: message.metadata.contentReferences
            } : {})
          }
        } : {}),
        ...(message.traceData ? {
          trace_data: {
            trace_id: message.traceData.traceId,
            transaction_id: message.traceData.transactionId
          }
        } : {})
      }))
    } : {})
  };
};
exports.transformToUpdateScheme = transformToUpdateScheme;