"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ProfilesManager = exports.ContextualProfileLevel = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _esQuery = require("@kbn/es-query");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _data_sources = require("../../common/data_sources");
var _add_log = require("../utils/add_log");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
/**
 * Options for the `getProfiles` method
 */
/**
 * Result of resolving the root profile
 */
let ContextualProfileLevel = exports.ContextualProfileLevel = /*#__PURE__*/function (ContextualProfileLevel) {
  ContextualProfileLevel["rootLevel"] = "rootLevel";
  ContextualProfileLevel["dataSourceLevel"] = "dataSourceLevel";
  ContextualProfileLevel["documentLevel"] = "documentLevel";
  return ContextualProfileLevel;
}({});
class ProfilesManager {
  constructor(rootProfileService, dataSourceProfileService, documentProfileService, ebtManager) {
    (0, _defineProperty2.default)(this, "rootContext$", void 0);
    (0, _defineProperty2.default)(this, "dataSourceContext$", void 0);
    (0, _defineProperty2.default)(this, "rootProfile", void 0);
    (0, _defineProperty2.default)(this, "dataSourceProfile", void 0);
    (0, _defineProperty2.default)(this, "prevRootProfileParams", void 0);
    (0, _defineProperty2.default)(this, "prevDataSourceProfileParams", void 0);
    (0, _defineProperty2.default)(this, "rootProfileAbortController", void 0);
    (0, _defineProperty2.default)(this, "dataSourceProfileAbortController", void 0);
    this.rootProfileService = rootProfileService;
    this.dataSourceProfileService = dataSourceProfileService;
    this.documentProfileService = documentProfileService;
    this.ebtManager = ebtManager;
    this.rootContext$ = new _rxjs.BehaviorSubject(rootProfileService.defaultContext);
    this.dataSourceContext$ = new _rxjs.BehaviorSubject(dataSourceProfileService.defaultContext);
    this.rootProfile = rootProfileService.getProfile({
      context: this.rootContext$.getValue()
    });
    this.dataSourceProfile = dataSourceProfileService.getProfile({
      context: this.dataSourceContext$.getValue()
    });
    this.rootContext$.pipe((0, _rxjs.skip)(1)).subscribe(context => {
      this.rootProfile = rootProfileService.getProfile({
        context
      });
    });
    this.dataSourceContext$.pipe((0, _rxjs.skip)(1)).subscribe(context => {
      this.dataSourceProfile = dataSourceProfileService.getProfile({
        context
      });
    });
  }

  /**
   * Resolves the root context profile
   * @param params The root profile provider parameters
   */
  async resolveRootProfile(params) {
    var _this$rootProfileAbor;
    const serializedParams = serializeRootProfileParams(params);
    if ((0, _lodash.isEqual)(this.prevRootProfileParams, serializedParams)) {
      return {
        getRenderAppWrapper: this.rootProfile.getRenderAppWrapper,
        getDefaultAdHocDataViews: this.rootProfile.getDefaultAdHocDataViews
      };
    }
    const abortController = new AbortController();
    (_this$rootProfileAbor = this.rootProfileAbortController) === null || _this$rootProfileAbor === void 0 ? void 0 : _this$rootProfileAbor.abort();
    this.rootProfileAbortController = abortController;
    let context = this.rootProfileService.defaultContext;
    try {
      context = await this.rootProfileService.resolve(params);
    } catch (e) {
      logResolutionError(ContextualProfileLevel.rootLevel, serializedParams, e);
    }
    if (abortController.signal.aborted) {
      return {
        getRenderAppWrapper: this.rootProfile.getRenderAppWrapper,
        getDefaultAdHocDataViews: this.rootProfile.getDefaultAdHocDataViews
      };
    }
    this.rootContext$.next(context);
    this.prevRootProfileParams = serializedParams;
    return {
      getRenderAppWrapper: this.rootProfile.getRenderAppWrapper,
      getDefaultAdHocDataViews: this.rootProfile.getDefaultAdHocDataViews
    };
  }

  /**
   * Resolves the data source context profile
   * @param params The data source profile provider parameters
   */
  async resolveDataSourceProfile(params) {
    var _this$dataSourceProfi;
    const serializedParams = serializeDataSourceProfileParams(params);
    if ((0, _lodash.isEqual)(this.prevDataSourceProfileParams, serializedParams)) {
      return;
    }
    const abortController = new AbortController();
    (_this$dataSourceProfi = this.dataSourceProfileAbortController) === null || _this$dataSourceProfi === void 0 ? void 0 : _this$dataSourceProfi.abort();
    this.dataSourceProfileAbortController = abortController;
    let context = this.dataSourceProfileService.defaultContext;
    try {
      context = await this.dataSourceProfileService.resolve({
        ...params,
        rootContext: this.rootContext$.getValue()
      });
    } catch (e) {
      logResolutionError(ContextualProfileLevel.dataSourceLevel, serializedParams, e);
    }
    if (abortController.signal.aborted) {
      return;
    }
    this.trackActiveProfiles(this.rootContext$.getValue().profileId, context.profileId);
    this.dataSourceContext$.next(context);
    this.prevDataSourceProfileParams = serializedParams;
  }

  /**
   * Resolves the document context profile for a given data table record
   * @param params The document profile provider parameters
   * @returns The data table record with a resolved document context
   */
  resolveDocumentProfile(params) {
    let context;
    return new Proxy(params.record, {
      has: (target, prop) => prop === 'context' || Reflect.has(target, prop),
      get: (target, prop, receiver) => {
        if (prop !== 'context') {
          return Reflect.get(target, prop, receiver);
        }
        if (!context) {
          try {
            context = this.documentProfileService.resolve({
              ...params,
              rootContext: this.rootContext$.getValue(),
              dataSourceContext: this.dataSourceContext$.getValue()
            });
          } catch (e) {
            logResolutionError(ContextualProfileLevel.documentLevel, {
              recordId: params.record.id
            }, e);
            context = this.documentProfileService.defaultContext;
          }
        }
        this.ebtManager.trackContextualProfileResolvedEvent({
          contextLevel: ContextualProfileLevel.documentLevel,
          profileId: context.profileId
        });
        return context;
      }
    });
  }

  /**
   * Retrieves an array of the resolved profiles
   * @param options Options for getting the profiles
   * @returns The resolved profiles
   */
  getProfiles({
    record
  } = {}) {
    return [this.rootProfile, this.dataSourceProfile, this.documentProfileService.getProfile({
      context: recordHasContext(record) ? record.context : this.documentProfileService.defaultContext
    })];
  }

  /**
   * Retrieves an observable of the resolved profiles that emits when the profiles change
   * @param options Options for getting the profiles
   * @returns The resolved profiles as an observable
   */
  getProfiles$(options = {}) {
    return (0, _rxjs.combineLatest)([this.rootContext$, this.dataSourceContext$]).pipe((0, _rxjs.map)(() => this.getProfiles(options)));
  }

  /**
   * Tracks the active profiles in the EBT context
   */
  trackActiveProfiles(rootContextProfileId, dataSourceContextProfileId) {
    const dscProfiles = [rootContextProfileId, dataSourceContextProfileId];
    this.ebtManager.trackContextualProfileResolvedEvent({
      contextLevel: ContextualProfileLevel.rootLevel,
      profileId: rootContextProfileId
    });
    this.ebtManager.trackContextualProfileResolvedEvent({
      contextLevel: ContextualProfileLevel.dataSourceLevel,
      profileId: dataSourceContextProfileId
    });
    this.ebtManager.updateProfilesContextWith(dscProfiles);
  }
}
exports.ProfilesManager = ProfilesManager;
const serializeRootProfileParams = params => {
  return {
    solutionNavId: params.solutionNavId
  };
};
const serializeDataSourceProfileParams = params => {
  return {
    dataViewId: (0, _data_sources.isDataSourceType)(params.dataSource, _data_sources.DataSourceType.DataView) ? params.dataSource.dataViewId : undefined,
    esqlQuery: (0, _data_sources.isDataSourceType)(params.dataSource, _data_sources.DataSourceType.Esql) && (0, _esQuery.isOfAggregateQueryType)(params.query) ? params.query.esql : undefined
  };
};
const recordHasContext = record => {
  return Boolean(record && 'context' in record);
};
const logResolutionError = (profileLevel, params, error) => {
  (0, _add_log.addLog)(`[ProfilesManager] ${profileLevel} context resolution failed with params: ${JSON.stringify(params, null, 2)}`, error);
};