"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getUsers = void 0;
var _lodash = require("lodash");
var _api = require("../../../common/types/api");
var _runtime_types = require("../../common/runtime_types");
var _authorization = require("../../authorization");
var _error = require("../../common/error");
var _utils = require("../cases/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getUsers = async ({
  caseId
}, casesClient, clientArgs) => {
  const {
    services: {
      userActionService
    },
    logger,
    authorization,
    securityStartPlugin
  } = clientArgs;
  try {
    // ensure that we have authorization for reading the case
    const theCase = await casesClient.cases.resolve({
      id: caseId,
      includeComments: false
    });
    const {
      participants,
      assignedAndUnassignedUsers
    } = await userActionService.getUsers({
      caseId
    });
    const entities = participants.map(participant => ({
      id: participant.id,
      owner: participant.owner
    }));
    await authorization.ensureAuthorized({
      entities,
      operation: _authorization.Operations.getUserActionUsers
    });
    const participantsUids = participants.filter(participant => participant.user.profile_uid != null).map(participant => participant.user.profile_uid);
    const assigneesUids = theCase.case.assignees.map(assignee => assignee.uid);
    const reporter = theCase.case.created_by;
    const reporterProfileIdAsArray = reporter.profile_uid != null ? [reporter.profile_uid] : [];
    const userProfileUids = new Set([...assignedAndUnassignedUsers, ...participantsUids, ...assigneesUids, ...reporterProfileIdAsArray].filter(uid => !(0, _lodash.isEmpty)(uid)));
    const userProfiles = await (0, _utils.getUserProfiles)(securityStartPlugin, userProfileUids, 'avatar');
    const participantsResponse = convertUserInfoToResponse(userProfiles, participants.map(participant => ({
      uid: participant.user.profile_uid,
      user: participant.user
    })));
    const assigneesResponse = convertUserInfoToResponse(userProfiles, theCase.case.assignees);
    const reporterResponse = convertUserInfoToResponse(userProfiles, [{
      uid: reporter.profile_uid,
      user: reporter
    }]);

    /**
     * To avoid duplicates, a user that is
     * a participant or an assignee or a reporter should not be
     *  part of the assignedAndUnassignedUsers Set
     */
    const unassignedUsers = removeAllFromSet(assignedAndUnassignedUsers, [...participantsUids, ...assigneesUids, ...reporterProfileIdAsArray]);
    const unassignedUsersResponse = convertUserInfoToResponse(userProfiles, [...unassignedUsers.values()].map(uid => ({
      uid
    })));
    const results = {
      participants: participantsResponse,
      assignees: assigneesResponse,
      unassignedUsers: unassignedUsersResponse,
      reporter: reporterResponse[0]
    };
    return (0, _runtime_types.decodeOrThrow)(_api.GetCaseUsersResponseRt)(results);
  } catch (error) {
    throw (0, _error.createCaseError)({
      message: `Failed to retrieve the case users case id: ${caseId}: ${error}`,
      error,
      logger
    });
  }
};
exports.getUsers = getUsers;
const convertUserInfoToResponse = (userProfiles, usersInfo) => {
  const response = [];
  for (const info of usersInfo) {
    response.push(getUserInformation(userProfiles, info.uid, info.user));
  }
  return response;
};
const getUserInformation = (userProfiles, uid, userInfo) => {
  var _ref, _userProfile$user$ema, _ref2, _userProfile$user$ful, _ref3, _userProfile$user$use, _ref4, _userProfile$uid;
  const userProfile = uid != null ? userProfiles.get(uid) : undefined;
  return {
    user: {
      email: (_ref = (_userProfile$user$ema = userProfile === null || userProfile === void 0 ? void 0 : userProfile.user.email) !== null && _userProfile$user$ema !== void 0 ? _userProfile$user$ema : userInfo === null || userInfo === void 0 ? void 0 : userInfo.email) !== null && _ref !== void 0 ? _ref : null,
      full_name: (_ref2 = (_userProfile$user$ful = userProfile === null || userProfile === void 0 ? void 0 : userProfile.user.full_name) !== null && _userProfile$user$ful !== void 0 ? _userProfile$user$ful : userInfo === null || userInfo === void 0 ? void 0 : userInfo.full_name) !== null && _ref2 !== void 0 ? _ref2 : null,
      username: (_ref3 = (_userProfile$user$use = userProfile === null || userProfile === void 0 ? void 0 : userProfile.user.username) !== null && _userProfile$user$use !== void 0 ? _userProfile$user$use : userInfo === null || userInfo === void 0 ? void 0 : userInfo.username) !== null && _ref3 !== void 0 ? _ref3 : null
    },
    avatar: getUserProfileAvatar(userProfile === null || userProfile === void 0 ? void 0 : userProfile.data.avatar),
    uid: (_ref4 = (_userProfile$uid = userProfile === null || userProfile === void 0 ? void 0 : userProfile.uid) !== null && _userProfile$uid !== void 0 ? _userProfile$uid : uid) !== null && _ref4 !== void 0 ? _ref4 : userInfo === null || userInfo === void 0 ? void 0 : userInfo.profile_uid
  };
};
const getUserProfileAvatar = avatar => {
  if (!avatar) {
    return avatar;
  }
  const res = {
    ...((0, _lodash.isString)(avatar.initials) ? {
      initials: avatar.initials
    } : {}),
    ...((0, _lodash.isString)(avatar.color) ? {
      color: avatar.color
    } : {}),
    ...((0, _lodash.isString)(avatar.imageUrl) ? {
      imageUrl: avatar.imageUrl
    } : {})
  };
  return res;
};
const removeAllFromSet = (originalSet, values) => {
  const newSet = new Set(originalSet);
  values.forEach(value => newSet.delete(value));
  return newSet;
};