"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runFleetSourcemapArtifactsMigration = runFleetSourcemapArtifactsMigration;
exports.scheduleSourceMapMigration = scheduleSourceMapMigration;
var _source_maps = require("../fleet/source_maps");
var _bulk_create_apm_source_maps = require("./bulk_create_apm_source_maps");
var _apm_system_index_constants = require("../settings/apm_indices/apm_system_index_constants");
var _create_apm_source_map_index_template = require("./create_apm_source_map_index_template");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const PER_PAGE = 10;
const TASK_ID = 'apm-source-map-migration-task-id';
const TASK_TYPE = 'apm-source-map-migration-task';
async function scheduleSourceMapMigration({
  coreStartPromise,
  pluginStartPromise,
  taskManager,
  logger
}) {
  if (!taskManager) {
    return;
  }
  logger.debug(`Register task "${TASK_TYPE}"`);
  taskManager.registerTaskDefinitions({
    [TASK_TYPE]: {
      title: 'Migrate fleet source map artifacts',
      description: `Migrates fleet source map artifacts to "${_apm_system_index_constants.APM_SOURCE_MAP_INDEX}" index`,
      timeout: '1h',
      maxAttempts: 5,
      createTaskRunner() {
        const taskState = {
          isAborted: false
        };
        return {
          async run() {
            logger.debug(`Run task: "${TASK_TYPE}"`);
            const coreStart = await coreStartPromise;
            const internalESClient = coreStart.elasticsearch.client.asInternalUser;

            // ensure that the index template has been created before running migration
            await (0, _create_apm_source_map_index_template.createApmSourceMapIndexTemplate)({
              client: internalESClient,
              logger
            });
            const pluginStart = await pluginStartPromise;
            const fleet = await pluginStart.fleet;
            if (fleet) {
              await runFleetSourcemapArtifactsMigration({
                taskState,
                fleet,
                internalESClient,
                logger
              });
            }
          },
          async cancel() {
            taskState.isAborted = true;
            logger.debug(`Task cancelled: "${TASK_TYPE}"`);
          }
        };
      }
    }
  });
  const pluginStart = await pluginStartPromise;
  const taskManagerStart = pluginStart.taskManager;
  if (taskManagerStart) {
    var _pluginStart$taskMana;
    logger.debug(`Task scheduled: "${TASK_TYPE}"`);
    await ((_pluginStart$taskMana = pluginStart.taskManager) === null || _pluginStart$taskMana === void 0 ? void 0 : _pluginStart$taskMana.ensureScheduled({
      id: TASK_ID,
      taskType: TASK_TYPE,
      scope: ['apm'],
      params: {},
      state: {}
    }));
  }
}
async function runFleetSourcemapArtifactsMigration({
  taskState,
  fleet,
  internalESClient,
  logger
}) {
  try {
    const latestApmSourceMapTimestamp = await getLatestApmSourceMap(internalESClient);
    const createdDateFilter = latestApmSourceMapTimestamp ? ` AND created:>${asLuceneEncoding(latestApmSourceMapTimestamp)}` : '';
    await paginateArtifacts({
      taskState,
      page: 1,
      apmArtifactClient: (0, _source_maps.getApmArtifactClient)(fleet),
      kuery: `type: sourcemap${createdDateFilter}`,
      logger,
      internalESClient
    });
  } catch (e) {
    logger.error('Failed to migrate APM fleet source map artifacts', {
      error: e
    });
  }
}

// will convert "2022-12-12T21:21:51.203Z" to "2022-12-12T21\:21\:51.203Z" because colons are not allowed when using Lucene syntax
function asLuceneEncoding(timestamp) {
  return timestamp.replaceAll(':', '\\:');
}
async function getArtifactsForPage({
  page,
  apmArtifactClient,
  kuery
}) {
  return await apmArtifactClient.listArtifacts({
    kuery,
    perPage: PER_PAGE,
    page,
    sortOrder: 'asc',
    sortField: 'created'
  });
}
async function paginateArtifacts({
  taskState,
  page,
  apmArtifactClient,
  kuery,
  logger,
  internalESClient
}) {
  if (taskState !== null && taskState !== void 0 && taskState.isAborted) {
    return;
  }
  const {
    total,
    items: artifacts
  } = await getArtifactsForPage({
    page,
    apmArtifactClient,
    kuery
  });
  if (artifacts.length === 0) {
    logger.debug('No source maps need to be migrated');
    return;
  }
  const migratedCount = (page - 1) * PER_PAGE + artifacts.length;
  logger.debug(`Migrating ${migratedCount} of ${total} source maps`);
  await (0, _bulk_create_apm_source_maps.bulkCreateApmSourceMaps)({
    artifacts,
    internalESClient
  });
  const hasMorePages = total > migratedCount;
  if (hasMorePages) {
    await paginateArtifacts({
      taskState,
      page: page + 1,
      apmArtifactClient,
      kuery,
      logger,
      internalESClient
    });
  } else {
    logger.debug(`Successfully migrated ${total} source maps`);
  }
}
async function getLatestApmSourceMap(internalESClient) {
  var _res$hits$hits$, _res$hits$hits$$_sour;
  const params = {
    index: _apm_system_index_constants.APM_SOURCE_MAP_INDEX,
    track_total_hits: false,
    size: 1,
    _source: ['created'],
    sort: [{
      created: {
        order: 'desc'
      }
    }],
    body: {
      query: {
        match_all: {}
      }
    }
  };
  const res = await internalESClient.search(params);
  return (_res$hits$hits$ = res.hits.hits[0]) === null || _res$hits$hits$ === void 0 ? void 0 : (_res$hits$hits$$_sour = _res$hits$hits$._source) === null || _res$hits$hits$$_sour === void 0 ? void 0 : _res$hits$hits$$_sour.created;
}