"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateMaintenanceWindow = updateMaintenanceWindow;
var _moment = _interopRequireDefault(require("moment"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _esQuery = require("@kbn/es-query");
var _common = require("../../../../../common");
var _get_es_query_config = require("../../../../lib/get_es_query_config");
var _lib = require("../../lib");
var _retry_if_conflicts = require("../../../../lib/retry_if_conflicts");
var _transforms = require("../../transforms");
var _maintenance_window = require("../../../../data/maintenance_window");
var _schemas = require("./schemas");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function updateMaintenanceWindow(context, params) {
  return await (0, _retry_if_conflicts.retryIfConflicts)(context.logger, `maintenanceWindowClient.update('${params.id})`, async () => {
    return await updateWithOCC(context, params);
  });
}
async function updateWithOCC(context, params) {
  const {
    savedObjectsClient,
    getModificationMetadata,
    logger,
    uiSettings
  } = context;
  const {
    id,
    data
  } = params;
  const {
    title,
    enabled,
    duration,
    rRule,
    categoryIds,
    scopedQuery
  } = data;
  const esQueryConfig = await (0, _get_es_query_config.getEsQueryConfig)(uiSettings);
  try {
    _schemas.updateMaintenanceWindowParamsSchema.validate(params);
  } catch (error) {
    throw _boom.default.badRequest(`Error validating update maintenance window data - ${error.message}`);
  }
  let scopedQueryWithGeneratedValue = scopedQuery;
  try {
    if (scopedQuery) {
      const dsl = JSON.stringify((0, _esQuery.buildEsQuery)(undefined, [{
        query: scopedQuery.kql,
        language: 'kuery'
      }], scopedQuery.filters, esQueryConfig));
      scopedQueryWithGeneratedValue = {
        ...scopedQuery,
        dsl
      };
    }
  } catch (error) {
    throw _boom.default.badRequest(`Error validating update maintenance window data - ${(0, _common.getScopedQueryErrorMessage)(error.message)}`);
  }
  try {
    const {
      attributes,
      id: fetchedId,
      version
    } = await (0, _maintenance_window.getMaintenanceWindowSo)({
      id,
      savedObjectsClient
    });
    const maintenanceWindow = (0, _transforms.transformMaintenanceWindowAttributesToMaintenanceWindow)({
      attributes,
      id: fetchedId
    });
    if (_moment.default.utc(maintenanceWindow.expirationDate).isBefore(new Date())) {
      throw _boom.default.badRequest('Cannot edit archived maintenance windows');
    }
    const expirationDate = (0, _lib.getMaintenanceWindowExpirationDate)({
      rRule: maintenanceWindow.rRule,
      duration: maintenanceWindow.duration
    });
    const modificationMetadata = await getModificationMetadata();
    let events = (0, _lib.generateMaintenanceWindowEvents)({
      rRule: rRule || maintenanceWindow.rRule,
      duration: typeof duration === 'number' ? duration : maintenanceWindow.duration,
      expirationDate
    });
    if (!(0, _lib.shouldRegenerateEvents)({
      maintenanceWindow,
      rRule,
      duration
    })) {
      events = (0, _lib.mergeEvents)({
        oldEvents: maintenanceWindow.events,
        newEvents: events
      });
    }
    const updateMaintenanceWindowAttributes = (0, _transforms.transformMaintenanceWindowToMaintenanceWindowAttributes)({
      ...maintenanceWindow,
      ...(title ? {
        title
      } : {}),
      ...(rRule ? {
        rRule: rRule
      } : {}),
      ...(categoryIds !== undefined ? {
        categoryIds
      } : {}),
      ...(scopedQueryWithGeneratedValue !== undefined ? {
        scopedQuery: scopedQueryWithGeneratedValue
      } : {}),
      ...(typeof duration === 'number' ? {
        duration
      } : {}),
      ...(typeof enabled === 'boolean' ? {
        enabled
      } : {}),
      expirationDate,
      events,
      updatedBy: modificationMetadata.updatedBy,
      updatedAt: modificationMetadata.updatedAt
    });

    // We are deleting and then creating rather than updating because SO.update
    // performs a partial update on the rRule, we would need to null out all of the fields
    // that are removed from a new rRule if that were the case.
    const result = await (0, _maintenance_window.createMaintenanceWindowSo)({
      savedObjectsClient,
      maintenanceWindowAttributes: updateMaintenanceWindowAttributes,
      savedObjectsCreateOptions: {
        id: fetchedId,
        version,
        overwrite: true
      }
    });
    return (0, _transforms.transformMaintenanceWindowAttributesToMaintenanceWindow)({
      attributes: result.attributes,
      id: result.id
    });
  } catch (e) {
    const errorMessage = `Failed to update maintenance window by id: ${id}, Error: ${e}`;
    logger.error(errorMessage);
    throw _boom.default.boomify(e, {
      message: errorMessage
    });
  }
}