"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteAlertsForQuery = void 0;
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _lib = require("../../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const PAGE_SIZE = 1000;
const MAX_ALERTS_TO_DELETE = 10000;
const deleteAlertsForQuery = async (context, indices, query, abortController) => {
  const esClient = await context.elasticsearchClientPromise;
  let numAlertsDeleted = 0;
  let pitId = null;
  let searchAfter = null;
  const errors = [];
  const taskIds = new Set();
  const alertUuidsToClear = [];
  do {
    if (!pitId) {
      const pitResponse = await esClient.openPointInTime({
        index: indices,
        keep_alive: '1m',
        ignore_unavailable: true
      });
      pitId = pitResponse.id;
    }
    try {
      // query for alerts to delete, sorted to return oldest first
      const searchResponse = await esClient.search({
        query,
        size: PAGE_SIZE,
        sort: [{
          [_ruleDataUtils.TIMESTAMP]: 'asc'
        }],
        pit: {
          id: pitId,
          keep_alive: '1m'
        },
        _source: [_ruleDataUtils.ALERT_RULE_UUID, _ruleDataUtils.SPACE_IDS, _ruleDataUtils.ALERT_INSTANCE_ID, _ruleDataUtils.TIMESTAMP],
        ...(searchAfter ? {
          search_after: searchAfter
        } : {})
      }, {
        signal: abortController.signal
      });
      if (searchResponse.hits.hits.length === 0) {
        searchAfter = null;
      } else {
        const numResults = searchResponse.hits.hits.length;
        searchAfter = searchResponse.hits.hits[numResults - 1].sort;

        // bulk delete the alert documents by id
        const bulkDeleteRequest = [];
        for (const alert of searchResponse.hits.hits) {
          bulkDeleteRequest.push({
            delete: {
              _index: alert._index,
              _id: alert._id
            }
          });
        }
        const bulkDeleteResponse = await esClient.bulk({
          operations: bulkDeleteRequest
        }, {
          signal: abortController.signal
        });

        // iterate and audit log each alert by ID
        bulkDeleteResponse.items.forEach((item, index) => {
          var _item$delete;
          const alertUuid = searchResponse.hits.hits[index]._id;
          if (((_item$delete = item.delete) === null || _item$delete === void 0 ? void 0 : _item$delete.result) === 'deleted') {
            var _searchResponse$hits$, _context$auditService;
            numAlertsDeleted++;
            const ruleId = (_searchResponse$hits$ = searchResponse.hits.hits[index]._source) === null || _searchResponse$hits$ === void 0 ? void 0 : _searchResponse$hits$[_ruleDataUtils.ALERT_RULE_UUID];
            if (ruleId) {
              taskIds.add(ruleId);
            }
            alertUuidsToClear.push(alertUuid);
            (_context$auditService = context.auditService) === null || _context$auditService === void 0 ? void 0 : _context$auditService.withoutRequest.log((0, _lib.alertAuditSystemEvent)({
              action: _lib.AlertAuditAction.DELETE,
              id: alertUuid,
              outcome: 'success'
            }));
          } else {
            var _context$auditService2, _item$delete2, _item$delete2$error, _item$delete3, _item$delete3$error;
            (_context$auditService2 = context.auditService) === null || _context$auditService2 === void 0 ? void 0 : _context$auditService2.withoutRequest.log((0, _lib.alertAuditSystemEvent)({
              action: _lib.AlertAuditAction.DELETE,
              id: alertUuid,
              outcome: 'failure',
              error: new Error((_item$delete2 = item.delete) === null || _item$delete2 === void 0 ? void 0 : (_item$delete2$error = _item$delete2.error) === null || _item$delete2$error === void 0 ? void 0 : _item$delete2$error.reason)
            }));
            errors.push(`Error deleting alert "${alertUuid}" - ${(_item$delete3 = item.delete) === null || _item$delete3 === void 0 ? void 0 : (_item$delete3$error = _item$delete3.error) === null || _item$delete3$error === void 0 ? void 0 : _item$delete3$error.reason}`);
          }
        });
      }
    } catch (err) {
      if (pitId) {
        await esClient.closePointInTime({
          id: pitId
        });
        pitId = null;
      }
      throw err;
    }
  } while (searchAfter != null && numAlertsDeleted < MAX_ALERTS_TO_DELETE);
  try {
    if (pitId) {
      await esClient.closePointInTime({
        id: pitId
      });
    }
  } catch (closeErr) {
    context.logger.error(`Failed to close point in time during alert deletion task: ${closeErr.message}`);
  }
  return {
    numAlertsDeleted,
    taskIds,
    alertUuidsToClear,
    errors
  };
};
exports.deleteAlertsForQuery = deleteAlertsForQuery;