"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchLogRateAnalysisForAlert = fetchLogRateAnalysisForAlert;
var _moment = _interopRequireDefault(require("moment"));
var _async = require("async");
var _lodash = require("lodash");
var _apmUtils = require("@kbn/apm-utils");
var _mlRandomSamplerUtils = require("@kbn/ml-random-sampler-utils");
var _get_log_rate_analysis_parameters_from_alert = require("../get_log_rate_analysis_parameters_from_alert");
var _get_swapped_window_parameters = require("../get_swapped_window_parameters");
var _get_log_rate_change = require("../get_log_rate_change");
var _get_baseline_and_deviation_rates = require("../get_baseline_and_deviation_rates");
var _get_log_rate_analysis_type_for_counts = require("../get_log_rate_analysis_type_for_counts");
var _log_rate_analysis_type = require("../log_rate_analysis_type");
var _fetch_index_info = require("./fetch_index_info");
var _fetch_significant_categories = require("./fetch_significant_categories");
var _fetch_significant_term_p_values = require("./fetch_significant_term_p_values");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_CONCURRENT_QUERIES = 5;
const CHUNK_SIZE = 50;
/**
 * Runs log rate analysis data on an index given some alert metadata.
 */
async function fetchLogRateAnalysisForAlert({
  esClient,
  abortSignal,
  arguments: args
}) {
  const {
    alertStartedAt,
    timefield = '@timestamp'
  } = args;
  const alertStart = (0, _moment.default)(alertStartedAt);
  const intervalFactor = (0, _get_log_rate_analysis_parameters_from_alert.getIntervalFactor)(args.alertRuleParameterTimeSize, args.alertRuleParameterTimeUnit);

  // The deviation time range is 1 lookback duration before the alert start.
  // The baseline time range is 2 lookback durations before the deviation time range.
  const windowParameters = {
    baselineMin: alertStart.clone().subtract(3 * intervalFactor, 'minutes').valueOf(),
    baselineMax: alertStart.clone().subtract(1 * intervalFactor, 'minutes').valueOf(),
    deviationMin: alertStart.clone().subtract(1 * intervalFactor, 'minutes').valueOf(),
    deviationMax: alertStart.valueOf()
  };
  const {
    searchQuery = {
      match_all: {}
    }
  } = args;

  // Step 1: Get field candidates and total doc counts.
  const indexInfoParams = {
    index: args.index,
    start: windowParameters.baselineMin,
    end: windowParameters.deviationMax,
    searchQuery: JSON.stringify(searchQuery),
    timeFieldName: timefield,
    ...windowParameters
  };
  const indexInfo = await (0, _apmUtils.withSpan)({
    name: 'fetch_index_info',
    type: 'aiops-log-rate-analysis-for-alert'
  }, () => (0, _fetch_index_info.fetchIndexInfo)({
    esClient,
    abortSignal,
    arguments: {
      ...indexInfoParams,
      textFieldCandidatesOverrides: ['message', 'error.message']
    }
  }));
  const {
    textFieldCandidates,
    keywordFieldCandidates
  } = indexInfo;
  const logRateAnalysisType = (0, _get_log_rate_analysis_type_for_counts.getLogRateAnalysisTypeForCounts)({
    baselineCount: indexInfo.baselineTotalDocCount,
    deviationCount: indexInfo.deviationTotalDocCount,
    windowParameters
  });

  // Just in case the log rate analysis type is 'dip', we need to swap
  // the window parameters for the analysis.
  const analysisWindowParameters = logRateAnalysisType === _log_rate_analysis_type.LOG_RATE_ANALYSIS_TYPE.SPIKE ? windowParameters : (0, _get_swapped_window_parameters.getSwappedWindowParameters)(windowParameters);

  // Step 2: Identify significant items.
  // The following code will fetch significant categories and term p-values
  // using an async queue. The field candidates will be passed on as chunks
  // of 50 fields with up to 5 concurrent queries. This is to prevent running
  // into bucket limit issues if we'd throw possibly hundreds of field candidates
  // into a single query.

  const significantItems = [];

  // Set up the queue: A queue item is an object with the function to call and
  // the field names to be passed to the function. This is done so we can push
  // queries for both keyword fields (using significant_terms/p-values) and
  // text fields (using categorize_text + custom code to identify significance)
  // into the same queue.
  const significantItemsQueue = (0, _async.queue)(async function ({
    fn,
    fieldNames
  }) {
    significantItems.push(...(await fn({
      esClient,
      abortSignal,
      arguments: {
        ...indexInfoParams,
        ...analysisWindowParameters,
        fieldNames,
        sampleProbability: (0, _mlRandomSamplerUtils.getSampleProbability)(indexInfo.deviationTotalDocCount + indexInfo.baselineTotalDocCount)
      }
    })));
  }, MAX_CONCURRENT_QUERIES);

  // Push the actual items to the queue. We don't need to chunk the text fields
  // since they are just `message` and `error.message`.
  significantItemsQueue.push([{
    fn: _fetch_significant_categories.fetchSignificantCategories,
    fieldNames: textFieldCandidates
  }, ...(0, _lodash.chunk)(keywordFieldCandidates, CHUNK_SIZE).map(fieldNames => ({
    fn: _fetch_significant_term_p_values.fetchSignificantTermPValues,
    fieldNames
  }))], err => {
    if (err) significantItemsQueue.kill();
  });

  // Wait for the queue to finish.
  await (0, _apmUtils.withSpan)({
    name: 'fetch_significant_items',
    type: 'aiops-log-rate-analysis-for-alert'
  }, () => significantItemsQueue.drain());

  // RETURN DATA
  // Adapt the raw significant items data for contextual insights.
  return {
    logRateAnalysisType,
    significantItems: significantItems.map(({
      fieldName,
      fieldValue,
      type,
      doc_count: docCount,
      bg_count: bgCount
    }) => {
      const {
        baselineBucketRate,
        deviationBucketRate
      } = (0, _get_baseline_and_deviation_rates.getBaselineAndDeviationRates)(logRateAnalysisType,
      // Normalize the amount of baseline buckets based on treating the
      // devation duration as 1 bucket.
      (windowParameters.baselineMax - windowParameters.baselineMin) / (windowParameters.deviationMax - windowParameters.deviationMin), 1, docCount, bgCount);
      const fieldType = type === 'keyword' ? 'metadata' : 'log message pattern';
      const data = {
        fieldType,
        fieldName,
        fieldValue: String(fieldValue).substring(0, 140),
        logRateChange: (0, _get_log_rate_change.getLogRateChange)(logRateAnalysisType, baselineBucketRate, deviationBucketRate).message
      };
      return {
        logRateChangeSort: bgCount > 0 ? docCount / bgCount : docCount,
        data
      };
    }).sort((a, b) => b.logRateChangeSort - a.logRateChangeSort).map(d => d.data)
  };
}