"use strict";
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ElasticV3ServerShipper = void 0;
var tslib_1 = require("tslib");
var node_fetch_1 = tslib_1.__importDefault(require("node-fetch"));
var rxjs_1 = require("rxjs");
var common_1 = require("../../common");
var SECOND = 1000;
var MINUTE = 60 * SECOND;
var HOUR = 60 * MINUTE;
var KIB = 1024;
var MAX_NUMBER_OF_EVENTS_IN_INTERNAL_QUEUE = 1000;
var MIN_TIME_SINCE_LAST_SEND = 10 * SECOND;
/**
 * Elastic V3 shipper to use on the server side.
 */
var ElasticV3ServerShipper = /** @class */ (function () {
    /**
     * Creates a new instance of the {@link ElasticV3ServerShipper}.
     * @param options {@link ElasticV3ShipperOptions}
     * @param initContext {@link AnalyticsClientInitContext}
     */
    function ElasticV3ServerShipper(options, initContext) {
        this.options = options;
        this.initContext = initContext;
        /** Observable to emit the stats of the processed events. */
        this.telemetryCounter$ = new rxjs_1.Subject();
        this.reportTelemetryCounters = (0, common_1.createTelemetryCounterHelper)(this.telemetryCounter$, ElasticV3ServerShipper.shipperName);
        this.internalQueue = [];
        this.shutdown$ = new rxjs_1.ReplaySubject(1);
        this.flush$ = new rxjs_1.Subject();
        this.inFlightRequests$ = new rxjs_1.BehaviorSubject(0);
        this.isOptedIn$ = new rxjs_1.BehaviorSubject(undefined);
        this.lastBatchSent = Date.now();
        this.clusterUuid = 'UNKNOWN';
        if (!options.buildShipperHeaders) {
            throw new Error('ElasticV3BrowserShipper requires options.buildShipperHeaders on initialization.');
        }
        if (!options.buildShipperUrl) {
            throw new Error('ElasticV3BrowserShipper requires options.buildShipperUrl on initialization.');
        }
        this.buildHeaders = options.buildShipperHeaders;
        this.url = options.buildShipperUrl({
            channelName: options.channelName,
        });
        this.setInternalSubscriber();
        this.checkConnectivity();
    }
    /**
     * Uses the `cluster_uuid` and `license_id` from the context to hold them in memory for the generation of the headers
     * used later on in the HTTP request.
     * @param newContext The full new context to set {@link EventContext}
     */
    ElasticV3ServerShipper.prototype.extendContext = function (newContext) {
        if (newContext.cluster_uuid) {
            this.clusterUuid = newContext.cluster_uuid;
        }
        if (newContext.license_id) {
            this.licenseId = newContext.license_id;
        }
    };
    /**
     * When `false`, it flushes the internal queue and stops sending events.
     * @param isOptedIn `true` for resume sending events. `false` to stop.
     */
    ElasticV3ServerShipper.prototype.optIn = function (isOptedIn) {
        this.isOptedIn$.next(isOptedIn);
        if (isOptedIn === false) {
            this.internalQueue.length = 0;
        }
    };
    /**
     * Enqueues the events to be sent via the leaky bucket algorithm.
     * @param events batched events {@link Event}
     */
    ElasticV3ServerShipper.prototype.reportEvents = function (events) {
        var _a;
        // If opted out OR offline for longer than 24 hours, skip processing any events.
        if (this.isOptedIn$.value === false || (this.firstTimeOffline && Date.now() - this.firstTimeOffline > 24 * HOUR)) {
            return;
        }
        var freeSpace = MAX_NUMBER_OF_EVENTS_IN_INTERNAL_QUEUE - this.internalQueue.length;
        // As per design, we only want store up-to 1000 events at a time. Drop anything that goes beyond that limit
        if (freeSpace < events.length) {
            var toDrop = events.length - freeSpace;
            var droppedEvents = events.splice(-toDrop, toDrop);
            this.reportTelemetryCounters(droppedEvents, {
                type: 'dropped',
                code: 'queue_full',
            });
        }
        (_a = this.internalQueue).push.apply(_a, tslib_1.__spreadArray([], tslib_1.__read(events), false));
    };
    /**
     * Triggers a flush of the internal queue to attempt to send any events held in the queue
     * and resolves the returned promise once the queue is emptied.
     */
    ElasticV3ServerShipper.prototype.flush = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var promise;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (this.flush$.isStopped) {
                            // If called after shutdown, return straight away
                            return [2 /*return*/];
                        }
                        promise = (0, rxjs_1.firstValueFrom)(this.inFlightRequests$.pipe((0, rxjs_1.skip)(1), // Skipping the first value because BehaviourSubjects always emit the current value on subscribe.
                        (0, rxjs_1.filter)(function (count) { return count === 0; }) // Wait until all the inflight requests are completed.
                        ));
                        this.flush$.next();
                        return [4 /*yield*/, promise];
                    case 1:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Shuts down the shipper.
     * Triggers a flush of the internal queue to attempt to send any events held in the queue.
     */
    ElasticV3ServerShipper.prototype.shutdown = function () {
        this.shutdown$.next();
        this.flush$.complete();
        this.shutdown$.complete();
        this.isOptedIn$.complete();
    };
    /**
     * Checks the server has connectivity to the remote endpoint.
     * The frequency of the connectivity tests will back off, starting with 1 minute, and multiplying by 2
     * until it reaches 1 hour. Then, it’ll keep the 1h frequency until it reaches 24h without connectivity.
     * At that point, it clears the queue and stops accepting events in the queue.
     * The connectivity checks will continue to happen every 1 hour just in case it regains it at any point.
     * @private
     */
    ElasticV3ServerShipper.prototype.checkConnectivity = function () {
        var _this = this;
        var backoff = 1 * MINUTE;
        (0, rxjs_1.merge)((0, rxjs_1.timer)(0, 1 * MINUTE), 
        // Also react to opt-in changes to avoid being stalled for 1 minute for the first connectivity check.
        // More details in: https://github.com/elastic/kibana/issues/135647
        this.isOptedIn$)
            .pipe((0, rxjs_1.takeUntil)(this.shutdown$), (0, rxjs_1.filter)(function () { return _this.isOptedIn$.value === true && _this.firstTimeOffline !== null; }), 
        // Using exhaustMap here because one request at a time is enough to check the connectivity.
        (0, rxjs_1.exhaustMap)(function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
            var ok;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, (0, node_fetch_1.default)(this.url, {
                            method: 'OPTIONS',
                        })];
                    case 1:
                        ok = (_a.sent()).ok;
                        if (!ok) {
                            throw new Error("Failed to connect to ".concat(this.url));
                        }
                        this.firstTimeOffline = null;
                        backoff = 1 * MINUTE;
                        return [2 /*return*/];
                }
            });
        }); }), (0, rxjs_1.retryWhen)(function (errors) {
            return errors.pipe((0, rxjs_1.takeUntil)(_this.shutdown$), (0, rxjs_1.tap)(function () {
                if (!_this.firstTimeOffline) {
                    _this.firstTimeOffline = Date.now();
                }
                else if (Date.now() - _this.firstTimeOffline > 24 * HOUR) {
                    _this.internalQueue.length = 0;
                }
                backoff = backoff * 2;
                if (backoff > 1 * HOUR) {
                    backoff = 1 * HOUR;
                }
            }), (0, rxjs_1.delayWhen)(function () { return (0, rxjs_1.timer)(backoff); }));
        }))
            .subscribe();
    };
    ElasticV3ServerShipper.prototype.setInternalSubscriber = function () {
        var _this = this;
        // Create an emitter that emits when MIN_TIME_SINCE_LAST_SEND have passed since the last time we sent the data
        var minimumTimeSinceLastSent$ = (0, rxjs_1.interval)(SECOND).pipe((0, rxjs_1.filter)(function () { return Date.now() - _this.lastBatchSent >= MIN_TIME_SINCE_LAST_SEND; }));
        (0, rxjs_1.merge)(minimumTimeSinceLastSent$.pipe((0, rxjs_1.takeUntil)(this.shutdown$), (0, rxjs_1.map)(function () { return ({ shouldFlush: false }); })), 
        // Whenever a `flush` request comes in
        this.flush$.pipe((0, rxjs_1.map)(function () { return ({ shouldFlush: true }); })), 
        // Attempt to send one last time on shutdown, flushing the queue
        this.shutdown$.pipe((0, rxjs_1.map)(function () { return ({ shouldFlush: true }); })))
            .pipe(
        // Only move ahead if it's opted-in and online, and there are some events in the queue
        (0, rxjs_1.filter)(function () {
            var shouldSendAnything = _this.isOptedIn$.value === true && _this.firstTimeOffline === null && _this.internalQueue.length > 0;
            // If it should not send anything, re-emit the inflight request observable just in case it's already 0
            if (!shouldSendAnything) {
                _this.inFlightRequests$.next(_this.inFlightRequests$.value);
            }
            return shouldSendAnything;
        }), 
        // Send the events:
        // 1. Set lastBatchSent and retrieve the events to send (clearing the queue) in a synchronous operation to avoid race conditions.
        (0, rxjs_1.map)(function (_a) {
            var shouldFlush = _a.shouldFlush;
            _this.lastBatchSent = Date.now();
            return _this.getEventsToSend(shouldFlush);
        }), 
        // 2. Skip empty buffers (just to be sure)
        (0, rxjs_1.filter)(function (events) { return events.length > 0; }), 
        // 3. Actually send the events
        // Using `mergeMap` here because we want to send events whenever the emitter says so:
        //   We don't want to skip emissions (exhaustMap) or enqueue them (concatMap).
        (0, rxjs_1.mergeMap)(function (eventsToSend) { return _this.sendEvents(eventsToSend); }))
            .subscribe();
    };
    /**
     * Calculates the size of the queue in bytes.
     * @returns The number of bytes held in the queue.
     * @private
     */
    ElasticV3ServerShipper.prototype.getQueueByteSize = function (queue) {
        var _this = this;
        return queue.reduce(function (acc, event) {
            return acc + _this.getEventSize(event);
        }, 0);
    };
    /**
     * Calculates the size of the event in bytes.
     * @param event The event to calculate the size of.
     * @returns The number of bytes held in the event.
     * @private
     */
    ElasticV3ServerShipper.prototype.getEventSize = function (event) {
        return Buffer.from(JSON.stringify(event)).length;
    };
    /**
     * Returns a queue of events of up-to 10kB. Or all events in the queue if it's a FLUSH action.
     * @remarks It mutates the internal queue by removing from it the events returned by this method.
     * @private
     */
    ElasticV3ServerShipper.prototype.getEventsToSend = function (shouldFlush) {
        // If the internal queue is already smaller than the minimum batch size, or it's a flush action, do a direct assignment.
        if (shouldFlush || this.getQueueByteSize(this.internalQueue) < 10 * KIB) {
            return this.internalQueue.splice(0, this.internalQueue.length);
        }
        // Otherwise, we'll feed the events to the leaky bucket queue until we reach 10kB.
        var queue = [];
        var queueByteSize = 0;
        while (queueByteSize < 10 * KIB) {
            var event_1 = this.internalQueue.shift();
            queueByteSize += this.getEventSize(event_1);
            queue.push(event_1);
        }
        return queue;
    };
    ElasticV3ServerShipper.prototype.sendEvents = function (events) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var code, error_1;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        this.initContext.logger.debug("Reporting ".concat(events.length, " events..."));
                        this.inFlightRequests$.next(this.inFlightRequests$.value + 1);
                        _a.label = 1;
                    case 1:
                        _a.trys.push([1, 3, , 4]);
                        return [4 /*yield*/, this.makeRequest(events)];
                    case 2:
                        code = _a.sent();
                        this.reportTelemetryCounters(events, { code: code });
                        this.initContext.logger.debug("Reported ".concat(events.length, " events..."));
                        return [3 /*break*/, 4];
                    case 3:
                        error_1 = _a.sent();
                        this.initContext.logger.debug("Failed to report ".concat(events.length, " events..."));
                        this.initContext.logger.debug(error_1);
                        this.reportTelemetryCounters(events, { code: error_1.code, error: error_1 });
                        this.firstTimeOffline = undefined;
                        return [3 /*break*/, 4];
                    case 4:
                        this.inFlightRequests$.next(Math.max(0, this.inFlightRequests$.value - 1));
                        return [2 /*return*/];
                }
            });
        });
    };
    ElasticV3ServerShipper.prototype.makeRequest = function (events) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var response, _a, _b, _c, _d, _e, _f, _g;
            return tslib_1.__generator(this, function (_h) {
                switch (_h.label) {
                    case 0: return [4 /*yield*/, (0, node_fetch_1.default)(this.url, tslib_1.__assign({ method: 'POST', body: (0, common_1.eventsToNDJSON)(events), headers: this.buildHeaders(this.clusterUuid, this.options.version, this.licenseId) }, (this.options.debug && { query: { debug: true } })))];
                    case 1:
                        response = _h.sent();
                        if (!this.options.debug) return [3 /*break*/, 3];
                        _b = (_a = this.initContext.logger).debug;
                        _d = (_c = "".concat(response.status, " - ")).concat;
                        return [4 /*yield*/, response.text()];
                    case 2:
                        _b.apply(_a, [_d.apply(_c, [_h.sent()])]);
                        _h.label = 3;
                    case 3:
                        if (!!response.ok) return [3 /*break*/, 5];
                        _e = common_1.ErrorWithCode.bind;
                        _g = (_f = "".concat(response.status, " - ")).concat;
                        return [4 /*yield*/, response.text()];
                    case 4: throw new (_e.apply(common_1.ErrorWithCode, [void 0, _g.apply(_f, [_h.sent()]), "".concat(response.status)]))();
                    case 5: return [2 /*return*/, "".concat(response.status)];
                }
            });
        });
    };
    /** Shipper's unique name */
    ElasticV3ServerShipper.shipperName = 'elastic_v3_server';
    return ElasticV3ServerShipper;
}());
exports.ElasticV3ServerShipper = ElasticV3ServerShipper;
