"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.HealthStatus = void 0;
exports.createAggregators = createAggregators;
exports.createMonitoringStatsStream = createMonitoringStatsStream;
exports.summarizeMonitoringStats = summarizeMonitoringStats;
var _rxjs = require("rxjs");
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _workload_statistics = require("./workload_statistics");
var _ephemeral_task_statistics = require("./ephemeral_task_statistics");
var _task_run_statistics = require("./task_run_statistics");
var _background_task_utilization_statistics = require("./background_task_utilization_statistics");
var _configuration_statistics = require("./configuration_statistics");
var _capacity_estimation = require("./capacity_estimation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
let HealthStatus = exports.HealthStatus = /*#__PURE__*/function (HealthStatus) {
  HealthStatus["Uninitialized"] = "uninitialized";
  HealthStatus["OK"] = "OK";
  HealthStatus["Warning"] = "warn";
  HealthStatus["Error"] = "error";
  return HealthStatus;
}({});
function createAggregators({
  taskStore,
  elasticsearchAndSOAvailability$,
  config,
  logger,
  taskDefinitions,
  adHocTaskCounter,
  startingCapacity,
  taskPollingLifecycle,
  ephemeralTaskLifecycle
}) {
  const aggregators = [(0, _configuration_statistics.createConfigurationAggregator)(config, startingCapacity, taskPollingLifecycle), (0, _workload_statistics.createWorkloadAggregator)({
    taskStore,
    elasticsearchAndSOAvailability$,
    refreshInterval: config.monitored_aggregated_stats_refresh_rate,
    pollInterval: config.poll_interval,
    logger,
    taskDefinitions
  })];
  if (taskPollingLifecycle) {
    aggregators.push((0, _task_run_statistics.createTaskRunAggregator)(taskPollingLifecycle, config.monitored_stats_running_average_window), (0, _background_task_utilization_statistics.createBackgroundTaskUtilizationAggregator)(taskPollingLifecycle, adHocTaskCounter, config.poll_interval, config.worker_utilization_running_average_window));
  }
  if (ephemeralTaskLifecycle && ephemeralTaskLifecycle.enabled) {
    aggregators.push((0, _ephemeral_task_statistics.createEphemeralTaskAggregator)(ephemeralTaskLifecycle, config.monitored_stats_running_average_window, startingCapacity));
  }
  return (0, _rxjs.merge)(...aggregators);
}
function createMonitoringStatsStream(provider$) {
  const initialStats = {
    last_update: new Date().toISOString(),
    stats: {}
  };
  return (0, _rxjs.merge)(
  // emit the initial stats
  (0, _rxjs.of)(initialStats),
  // emit updated stats whenever a provider updates a specific key on the stats
  provider$.pipe((0, _rxjs.map)(({
    key,
    value
  }) => {
    return {
      value: {
        timestamp: new Date().toISOString(),
        value
      },
      key
    };
  }), (0, _rxjs.scan)((monitoringStats, {
    key,
    value
  }) => {
    // incrementally merge stats as they come in
    (0, _saferLodashSet.set)(monitoringStats.stats, key, value);
    monitoringStats.last_update = new Date().toISOString();
    return monitoringStats;
  }, initialStats)));
}
function summarizeMonitoringStats(logger, {
  last_update,
  stats: {
    runtime,
    workload,
    configuration,
    ephemeral,
    utilization
  }
}, config, assumedKibanaInstances) {
  const summarizedStats = (0, _capacity_estimation.withCapacityEstimate)(logger, {
    ...(configuration ? {
      configuration: {
        ...configuration,
        status: HealthStatus.OK
      }
    } : {}),
    ...(runtime ? {
      runtime: {
        timestamp: runtime.timestamp,
        ...(0, _task_run_statistics.summarizeTaskRunStat)(logger, runtime.value, config)
      }
    } : {}),
    ...(workload ? {
      workload: {
        timestamp: workload.timestamp,
        ...(0, _workload_statistics.summarizeWorkloadStat)(workload.value)
      }
    } : {}),
    ...(ephemeral ? {
      ephemeral: {
        timestamp: ephemeral.timestamp,
        ...(0, _ephemeral_task_statistics.summarizeEphemeralStat)(ephemeral.value)
      }
    } : {})
  }, assumedKibanaInstances);
  return {
    last_update,
    stats: summarizedStats
  };
}