"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateDelayBasedOnAttempts = calculateDelayBasedOnAttempts;
exports.getRetryAt = getRetryAt;
exports.getRetryDate = getRetryDate;
exports.getTimeout = getTimeout;
var _lodash = require("lodash");
var _task = require("../task");
var _task_running = require("../task_running");
var _intervals = require("./intervals");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getRetryAt(task, taskDefinition) {
  const taskTimeout = getTimeout(task, taskDefinition);
  if (task.schedule) {
    return (0, _intervals.maxIntervalFromDate)(new Date(), task.schedule.interval, taskTimeout);
  }
  return getRetryDate({
    attempts: task.attempts + 1,
    // Fake an error. This allows retry logic when tasks keep timing out
    // and lets us set a proper "retryAt" value each time.
    error: new Error('Task timeout'),
    addDuration: taskTimeout
  });
}
function getRetryDate({
  error,
  attempts,
  addDuration
}) {
  var _isRetryableError;
  const retry = (_isRetryableError = (0, _task_running.isRetryableError)(error)) !== null && _isRetryableError !== void 0 ? _isRetryableError : true;
  let result;
  if (retry instanceof Date) {
    result = retry;
  } else if (retry === true) {
    result = new Date(Date.now() + calculateDelayBasedOnAttempts(attempts));
  }

  // Add a duration to the result
  if (addDuration && result) {
    result = (0, _intervals.intervalFromDate)(result, addDuration);
  }
  return result;
}
function calculateDelayBasedOnAttempts(attempts) {
  // Return 30s for the first retry attempt
  if (attempts === 1) {
    return 30 * 1000;
  } else {
    const defaultBackoffPerFailure = 5 * 60 * 1000;
    const maxDelay = 60 * 60 * 1000;
    // For each remaining attempt return an exponential delay with jitter that is capped at 1 hour.
    // We adjust the attempts by 2 to ensure that delay starts at 5m for the second retry attempt
    // and increases exponentially from there.
    return (0, _lodash.random)(Math.min(maxDelay, defaultBackoffPerFailure * Math.pow(2, attempts - 2)));
  }
}
function getTimeout(task, taskDefinition) {
  var _taskDefinition$timeo2;
  if (task.schedule) {
    var _taskDefinition$timeo;
    return (_taskDefinition$timeo = taskDefinition === null || taskDefinition === void 0 ? void 0 : taskDefinition.timeout) !== null && _taskDefinition$timeo !== void 0 ? _taskDefinition$timeo : _task.DEFAULT_TIMEOUT;
  }
  return task.timeoutOverride ? task.timeoutOverride : (_taskDefinition$timeo2 = taskDefinition === null || taskDefinition === void 0 ? void 0 : taskDefinition.timeout) !== null && _taskDefinition$timeo2 !== void 0 ? _taskDefinition$timeo2 : _task.DEFAULT_TIMEOUT;
}