"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runTaskPerPrivateLocation = exports.runSynPrivateLocationMonitorsTaskSoon = exports.resetSyncPrivateCleanUpState = exports.disableSyncPrivateLocationTask = exports.SyncPrivateLocationMonitorsTask = exports.PRIVATE_LOCATIONS_SYNC_TASK_ID = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("@kbn/spaces-plugin/common/constants");
var _moment = _interopRequireDefault(require("moment"));
var _common = require("@kbn/alerting-plugin/common");
var _pRetry = _interopRequireDefault(require("p-retry"));
var _saved_objects = require("../../common/types/saved_objects");
var _deploy_private_location_monitors = require("./deploy_private_location_monitors");
var _clean_up_duplicate_policies = require("./clean_up_duplicate_policies");
var _get_private_locations = require("../synthetics_service/get_private_locations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TASK_TYPE = 'Synthetics:Sync-Private-Location-Monitors';
const PRIVATE_LOCATIONS_SYNC_TASK_ID = exports.PRIVATE_LOCATIONS_SYNC_TASK_ID = `${TASK_TYPE}-single-instance`;
const TASK_SCHEDULE = '60m';
class SyncPrivateLocationMonitorsTask {
  constructor(serverSetup, syntheticsMonitorClient) {
    (0, _defineProperty2.default)(this, "deployPackagePolicies", void 0);
    (0, _defineProperty2.default)(this, "start", async () => {
      const {
        pluginsStart: {
          taskManager
        }
      } = this.serverSetup;
      this.debugLog(`Scheduling private location task`);
      await taskManager.ensureScheduled({
        id: PRIVATE_LOCATIONS_SYNC_TASK_ID,
        state: {},
        schedule: {
          interval: TASK_SCHEDULE
        },
        taskType: TASK_TYPE,
        params: {}
      });
      this.debugLog(`Sync private location monitors task scheduled successfully`);
    });
    (0, _defineProperty2.default)(this, "debugLog", message => {
      this.serverSetup.logger.debug(`[SyncPrivateLocationMonitorsTask] ${message}`);
    });
    this.serverSetup = serverSetup;
    this.syntheticsMonitorClient = syntheticsMonitorClient;
    this.deployPackagePolicies = new _deploy_private_location_monitors.DeployPrivateLocationMonitors(serverSetup, syntheticsMonitorClient);
  }
  registerTaskDefinition(taskManager) {
    taskManager.registerTaskDefinitions({
      [TASK_TYPE]: {
        title: 'Synthetics Sync Global Params Task',
        description: 'This task is executed so that we can sync private location monitors for example when global params are updated',
        timeout: '10m',
        maxAttempts: 1,
        createTaskRunner: ({
          taskInstance
        }) => {
          return {
            run: async () => {
              return this.runTask({
                taskInstance
              });
            }
          };
        }
      }
    });
  }
  async runTask({
    taskInstance
  }) {
    this.debugLog(`Syncing private location monitors, current task state is ${JSON.stringify(taskInstance.state)}`);
    const {
      coreStart: {
        savedObjects
      },
      logger,
      encryptedSavedObjects
    } = this.serverSetup;
    let lastStartedAt = taskInstance.state.lastStartedAt;
    // if it's too old, set it to 10 minutes ago to avoid syncing everything the first time
    if (!lastStartedAt || (0, _moment.default)(lastStartedAt).isBefore((0, _moment.default)().subtract(6, 'hour'))) {
      lastStartedAt = (0, _moment.default)().subtract(10, 'minute').toISOString();
    }
    const taskState = this.getNewTaskState({
      taskInstance
    });
    try {
      const soClient = savedObjects.createInternalRepository([_common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE]);
      const allPrivateLocations = await (0, _get_private_locations.getPrivateLocations)(soClient, _constants.ALL_SPACES_ID);
      if (taskInstance.state.privateLocationId) {
        // if privateLocationId exists on state, we just perform sync and exit
        await this.deployPackagePolicies.syncAllPackagePolicies({
          allPrivateLocations,
          encryptedSavedObjects,
          privateLocationId: taskInstance.state.privateLocationId,
          soClient: savedObjects.createInternalRepository()
        });
        return {
          state: {
            ...taskInstance.state,
            privateLocationId: undefined
          }
        };
      }
      const defaultState = {
        state: taskState,
        schedule: {
          interval: TASK_SCHEDULE
        }
      };
      const {
        performCleanupSync
      } = await this.cleanUpDuplicatedPackagePolicies(soClient, taskState);
      if (allPrivateLocations.length === 0) {
        this.debugLog(`No private locations found, skipping sync of private location monitors`);
        return {
          state: taskState,
          schedule: {
            interval: TASK_SCHEDULE
          }
        };
      }
      if (performCleanupSync) {
        this.debugLog(`Syncing private location monitors because cleanup performed a change`);
        if (allPrivateLocations.length > 1) {
          // if there are multiple locations, we run a task per location to optimize it
          for (const location of allPrivateLocations) {
            await runTaskPerPrivateLocation({
              server: this.serverSetup,
              privateLocationId: location.id
            });
          }
        } else {
          await this.deployPackagePolicies.syncAllPackagePolicies({
            allPrivateLocations,
            soClient,
            encryptedSavedObjects
          });
        }
        return defaultState;
      }
      if (taskState.disableAutoSync) {
        this.debugLog(`Auto sync is disabled, skipping sync of private location monitors`);
        return defaultState;
      }
      const monitorMwsIds = await this.fetchMonitorMwsIds(soClient);
      if (monitorMwsIds.length === 0) {
        this.debugLog(`No monitors with maintenance windows found, skipping sync of private location monitors`);
        return defaultState;
      }
      const {
        hasMWsChanged,
        updatedMWs,
        missingMWIds,
        maintenanceWindows
      } = await this.hasMWsChanged({
        soClient,
        taskState,
        lastStartedAt,
        monitorMwsIds
      });
      const dataChangeSync = hasMWsChanged && !taskState.disableAutoSync;
      if (dataChangeSync) {
        this.debugLog(`Syncing private location monitors because data has changed`);
        await this.deployPackagePolicies.syncPackagePoliciesForMws({
          allPrivateLocations,
          soClient,
          updatedMWs,
          missingMWIds,
          // this is passed so we don't have to fetch them again in the method
          maintenanceWindows
        });
        this.debugLog(`Sync of private location monitors succeeded`);
      } else {
        if (taskState.disableAutoSync) {
          this.debugLog(`Auto sync is disabled, skipping sync of private location monitors`);
        } else {
          this.debugLog(`No data has changed since last run ${lastStartedAt}, skipping sync of private location monitors`);
        }
      }
    } catch (error) {
      logger.error(`Sync of private location monitors failed: ${error.message}`);
      return {
        error,
        state: taskState,
        schedule: {
          interval: TASK_SCHEDULE
        }
      };
    }
    return {
      state: taskState,
      schedule: {
        interval: TASK_SCHEDULE
      }
    };
  }
  getNewTaskState({
    taskInstance
  }) {
    var _taskInstance$state$d;
    const startedAt = taskInstance.startedAt || new Date();
    return {
      lastStartedAt: startedAt.toISOString(),
      hasAlreadyDoneCleanup: taskInstance.state.hasAlreadyDoneCleanup || false,
      maxCleanUpRetries: taskInstance.state.maxCleanUpRetries || 3,
      disableAutoSync: (_taskInstance$state$d = taskInstance.state.disableAutoSync) !== null && _taskInstance$state$d !== void 0 ? _taskInstance$state$d : false
    };
  }
  parseLocations(config) {
    const {
      locations
    } = config;
    const privateLocations = locations.filter(loc => !loc.isServiceManaged);
    const publicLocations = locations.filter(loc => loc.isServiceManaged);
    return {
      privateLocations,
      publicLocations
    };
  }
  async fetchMonitorMwsIds(soClient) {
    const monitorsWithMws = await soClient.find({
      type: _saved_objects.syntheticsMonitorSOTypes,
      perPage: 0,
      namespaces: [_constants.ALL_SPACES_ID],
      fields: [],
      aggs: {
        monitorMws: {
          terms: {
            field: `${_saved_objects.syntheticsMonitorAttributes}.maintenance_windows`,
            size: 1000
          }
        },
        legacyMonitorsMws: {
          terms: {
            field: `${_saved_objects.legacyMonitorAttributes}.maintenance_windows`,
            size: 1000
          }
        }
      }
    });
    const {
      monitorMws,
      legacyMonitorsMws
    } = monitorsWithMws.aggregations || {};
    const monitorMwsIds = (monitorMws === null || monitorMws === void 0 ? void 0 : monitorMws.buckets.map(b => b.key)) || [];
    const legacyMonitorMwsIds = (legacyMonitorsMws === null || legacyMonitorsMws === void 0 ? void 0 : legacyMonitorsMws.buckets.map(b => b.key)) || [];
    this.debugLog(`Fetched monitor MWs IDs: ${JSON.stringify(monitorMwsIds)}`);
    this.debugLog(`Fetched legacy monitor MWs IDs: ${JSON.stringify(legacyMonitorMwsIds)}`);
    return Array.from(new Set([...monitorMwsIds, ...legacyMonitorMwsIds]));
  }
  async hasMWsChanged({
    lastStartedAt,
    monitorMwsIds
  }) {
    var _await$syntheticsServ;
    const {
      syntheticsService
    } = this.syntheticsMonitorClient;
    const maintenanceWindows = (_await$syntheticsServ = await syntheticsService.getMaintenanceWindows(_constants.ALL_SPACES_ID)) !== null && _await$syntheticsServ !== void 0 ? _await$syntheticsServ : [];
    // check if any of the MWs were updated since the last run
    const updatedMWs = maintenanceWindows.filter(mw => {
      const updatedAt = mw.updatedAt;
      return (0, _moment.default)(updatedAt).isAfter((0, _moment.default)(lastStartedAt));
    });
    this.debugLog(`Updated MWs: ${updatedMWs.map(mw => mw.id).join(', ')}`);

    // check if any MWs are missing
    const missingMWIds = monitorMwsIds.filter(mwId => {
      return !maintenanceWindows.find(mw => mw.id === mwId);
    });
    this.debugLog('Missing MW IDs: ' + JSON.stringify(missingMWIds));
    return {
      hasMWsChanged: updatedMWs.length > 0 || missingMWIds.length > 0,
      updatedMWs,
      missingMWIds,
      maintenanceWindows: maintenanceWindows.filter(mw => monitorMwsIds.includes(mw.id))
    };
  }
  async cleanUpDuplicatedPackagePolicies(soClient, taskState) {
    return await (0, _clean_up_duplicate_policies.cleanUpDuplicatedPackagePolicies)(this.serverSetup, soClient, taskState);
  }
}
exports.SyncPrivateLocationMonitorsTask = SyncPrivateLocationMonitorsTask;
const runSynPrivateLocationMonitorsTaskSoon = async ({
  server,
  retries = 5
}) => {
  try {
    await (0, _pRetry.default)(async () => {
      const {
        logger,
        pluginsStart: {
          taskManager
        }
      } = server;
      logger.debug(`Scheduling Synthetics sync private location monitors task soon`);
      await taskManager.runSoon(PRIVATE_LOCATIONS_SYNC_TASK_ID);
      logger.debug(`Synthetics sync private location task scheduled successfully`);
    }, {
      retries
    });
  } catch (error) {
    server.logger.error(`Error scheduling Synthetics sync private location monitors task: ${error.message}`, {
      error
    });
  }
};
exports.runSynPrivateLocationMonitorsTaskSoon = runSynPrivateLocationMonitorsTaskSoon;
const resetSyncPrivateCleanUpState = async ({
  server,
  hasAlreadyDoneCleanup = false
}) => {
  const {
    logger,
    pluginsStart: {
      taskManager
    }
  } = server;
  logger.debug(`Resetting Synthetics sync private location monitors cleanup state`);
  await taskManager.bulkUpdateState([PRIVATE_LOCATIONS_SYNC_TASK_ID], state => ({
    ...state,
    hasAlreadyDoneCleanup
  }));
  await runSynPrivateLocationMonitorsTaskSoon({
    server
  });
  logger.debug(`Synthetics sync private location monitors cleanup state reset successfully`);
};
exports.resetSyncPrivateCleanUpState = resetSyncPrivateCleanUpState;
const disableSyncPrivateLocationTask = async ({
  server,
  disableAutoSync
}) => {
  const {
    logger,
    pluginsStart: {
      taskManager
    }
  } = server;
  logger.debug(`Setting Synthetics sync private location monitors disableAutoSync to ${disableAutoSync}`);
  await taskManager.bulkUpdateState([PRIVATE_LOCATIONS_SYNC_TASK_ID], state => ({
    ...state,
    disableAutoSync
  }));
  logger.debug(`Synthetics sync private location monitors disableAutoSync set successfully`);
};
exports.disableSyncPrivateLocationTask = disableSyncPrivateLocationTask;
const runTaskPerPrivateLocation = async ({
  server,
  privateLocationId
}) => {
  const {
    pluginsStart: {
      taskManager
    }
  } = server;
  await taskManager.ensureScheduled({
    id: `${TASK_TYPE}:${privateLocationId}`,
    params: {},
    taskType: TASK_TYPE,
    runAt: new Date(Date.now() + 3 * 1000),
    state: {
      privateLocationId
    }
  });
};
exports.runTaskPerPrivateLocation = runTaskPerPrivateLocation;