"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DeployPrivateLocationMonitors = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("@kbn/spaces-plugin/common/constants");
var _saved_objects = require("../../common/types/saved_objects");
var _utils = require("../synthetics_service/utils");
var _runtime_types = require("../../common/runtime_types");
var _monitor_config_repository = require("../services/monitor_config_repository");
var _format_configs = require("../synthetics_service/formatters/public_formatters/format_configs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DeployPrivateLocationMonitors {
  constructor(serverSetup, syntheticsMonitorClient) {
    (0, _defineProperty2.default)(this, "debugLog", message => {
      this.serverSetup.logger.debug(`[DeployPrivateLocationMonitors] ${message}`);
    });
    (0, _defineProperty2.default)(this, "removeMwsFromMonitorConfigs", async ({
      mwId,
      monitors,
      soClient
    }) => {
      var _monitors$length;
      this.debugLog(`Removing maintenance window id: ${mwId} from monitors count: ${(_monitors$length = monitors === null || monitors === void 0 ? void 0 : monitors.length) !== null && _monitors$length !== void 0 ? _monitors$length : 0}`);
      const toUpdateMonitors = monitors.map(monitor => {
        var _monitor$namespaces;
        const existingMws = monitor.attributes[_runtime_types.ConfigKey.MAINTENANCE_WINDOWS] || [];
        const updatedMws = existingMws.filter(id => id !== mwId);
        return {
          id: monitor.id,
          type: monitor.type,
          attributes: {
            [_runtime_types.ConfigKey.MAINTENANCE_WINDOWS]: updatedMws
          },
          namespace: (_monitor$namespaces = monitor.namespaces) === null || _monitor$namespaces === void 0 ? void 0 : _monitor$namespaces[0]
        };
      });
      const result = await soClient.bulkUpdate(toUpdateMonitors);
      this.debugLog(`Removed maintenance window id: ${mwId} from monitors, updated monitors count: ${result.saved_objects.length}`);
    });
    this.serverSetup = serverSetup;
    this.syntheticsMonitorClient = syntheticsMonitorClient;
  }
  async syncPackagePoliciesForMws({
    allPrivateLocations,
    updatedMWs,
    missingMWIds,
    soClient,
    maintenanceWindows
  }) {
    if (allPrivateLocations.length === 0) {
      this.debugLog('No private locations found, skipping sync of private location monitors');
      return;
    }
    const {
      syntheticsService
    } = this.syntheticsMonitorClient;
    const paramsBySpace = await syntheticsService.getSyntheticsParams({
      spaceId: _constants.ALL_SPACES_ID
    });

    // config can have multiple mws, so we need to track which configs have been updated to avoid duplicate processing
    const listOfUpdatedConfigs = [];
    return this.serverSetup.fleet.runWithCache(async () => {
      const commonProps = {
        listOfUpdatedConfigs,
        allPrivateLocations,
        maintenanceWindows,
        soClient,
        paramsBySpace
      };
      for (const mw of updatedMWs || []) {
        await this.updateMonitorsForMw({
          ...commonProps,
          mwId: mw.id
        });
      }
      for (const mwId of missingMWIds || []) {
        await this.updateMonitorsForMw({
          ...commonProps,
          mwId,
          isMissingMw: true
        });
      }
    });
  }
  async updateMonitorsForMw({
    listOfUpdatedConfigs,
    allPrivateLocations,
    mwId,
    maintenanceWindows,
    soClient,
    paramsBySpace,
    isMissingMw = false
  }) {
    const {
      pluginsStart: {
        encryptedSavedObjects
      }
    } = this.serverSetup;
    const encryptedClient = encryptedSavedObjects.getClient();
    const finder = await encryptedClient.createPointInTimeFinderDecryptedAsInternalUser({
      type: _saved_objects.syntheticsMonitorSOTypes,
      perPage: 500,
      namespaces: ['*'],
      // TODO: Exclude public monitors
      filter: `${_saved_objects.syntheticsMonitorAttributes}.${_runtime_types.ConfigKey.MAINTENANCE_WINDOWS}: "${mwId}" or ${_saved_objects.legacyMonitorAttributes}.${_runtime_types.ConfigKey.MAINTENANCE_WINDOWS}: "${mwId}"`
    });
    for await (const result of finder.find()) {
      var _monitors$length2;
      const monitors = result.saved_objects.filter(monitor => {
        // Avoid processing the same config multiple times, updating it once will update all mws on it
        if (listOfUpdatedConfigs.includes(monitor.id)) {
          this.debugLog(`Skipping monitor id: ${monitor.id} as it has already been processed for another maintenance window`);
          return false;
        }
        listOfUpdatedConfigs.push(monitor.id);
        return true;
      });
      this.debugLog(`Processing mw id: ${mwId}, monitors count: ${(_monitors$length2 = monitors === null || monitors === void 0 ? void 0 : monitors.length) !== null && _monitors$length2 !== void 0 ? _monitors$length2 : 0}`);
      const {
        configsBySpaces,
        monitorSpaceIds
      } = this.mixParamsWithMonitors(monitors, paramsBySpace);
      await this.deployEditMonitors({
        allPrivateLocations,
        configsBySpaces,
        monitorSpaceIds,
        paramsBySpace,
        maintenanceWindows
      });
      if (isMissingMw) {
        await this.removeMwsFromMonitorConfigs({
          mwId,
          monitors,
          soClient
        });
      }
    }
    finder.close().catch(() => {});
    this.debugLog(`Syncing package policies for updated maintenance window id: ${mwId}`);
  }
  async syncAllPackagePolicies({
    allPrivateLocations,
    encryptedSavedObjects,
    soClient,
    spaceIdToSync,
    privateLocationId
  }) {
    if (allPrivateLocations.length === 0) {
      this.debugLog('No private locations found, skipping sync of private location monitors');
      return;
    }
    const {
      configsBySpaces,
      paramsBySpace,
      monitorSpaceIds,
      maintenanceWindows
    } = await this.getAllMonitorConfigs({
      encryptedSavedObjects,
      soClient,
      privateLocationId,
      spaceId: spaceIdToSync
    });
    return this.serverSetup.fleet.runWithCache(async () => {
      this.debugLog(`Starting sync of private location monitors for spaces: ${Array.from(monitorSpaceIds).join(', ')}`);
      await this.deployEditMonitors({
        allPrivateLocations: allPrivateLocations.filter(loc => loc.id === privateLocationId || !privateLocationId),
        configsBySpaces,
        monitorSpaceIds,
        paramsBySpace,
        maintenanceWindows
      });
      this.debugLog('Completed sync of private location monitors');
    });
  }
  async deployEditMonitors({
    allPrivateLocations,
    configsBySpaces,
    monitorSpaceIds,
    paramsBySpace,
    maintenanceWindows
  }) {
    const {
      privateLocationAPI
    } = this.syntheticsMonitorClient;
    for (const spaceId of monitorSpaceIds) {
      var _monitors$length3;
      const privateConfigs = [];
      const monitors = configsBySpaces[spaceId];
      this.debugLog(`Processing spaceId: ${spaceId}, monitors count: ${(_monitors$length3 = monitors === null || monitors === void 0 ? void 0 : monitors.length) !== null && _monitors$length3 !== void 0 ? _monitors$length3 : 0}`);
      if (!monitors) {
        continue;
      }
      for (const monitor of monitors) {
        const {
          privateLocations
        } = this.parseLocations(monitor);
        if (privateLocations.length > 0) {
          privateConfigs.push({
            config: monitor,
            globalParams: paramsBySpace[spaceId]
          });
        }
      }
      if (privateConfigs.length > 0) {
        this.debugLog(`Syncing private configs for spaceId: ${spaceId}, privateConfigs count: ${privateConfigs.length}`);
        await privateLocationAPI.editMonitors(privateConfigs, allPrivateLocations, spaceId, maintenanceWindows);
      } else {
        this.debugLog(`No privateConfigs to sync for spaceId: ${spaceId}`);
      }
    }
  }
  async getAllMonitorConfigs({
    soClient,
    encryptedSavedObjects,
    spaceId = _constants.ALL_SPACES_ID,
    privateLocationId
  }) {
    const {
      syntheticsService
    } = this.syntheticsMonitorClient;
    const paramsBySpacePromise = syntheticsService.getSyntheticsParams({
      spaceId
    });
    const maintenanceWindowsPromise = syntheticsService.getMaintenanceWindows(spaceId);
    const monitorConfigRepository = new _monitor_config_repository.MonitorConfigRepository(soClient, encryptedSavedObjects.getClient());
    const monitorsPromise = monitorConfigRepository.findDecryptedMonitors({
      spaceId,
      ...(privateLocationId && {
        filter: `${_saved_objects.syntheticsMonitorAttributes}.locations.id:"${privateLocationId}"`
      })
    });
    const [paramsBySpace, monitors, maintenanceWindows] = await Promise.all([paramsBySpacePromise, monitorsPromise, maintenanceWindowsPromise]);
    return {
      ...this.mixParamsWithMonitors(monitors, paramsBySpace),
      paramsBySpace,
      maintenanceWindows
    };
  }
  parseLocations(config) {
    const {
      locations
    } = config;
    const privateLocations = locations.filter(loc => !loc.isServiceManaged);
    const publicLocations = locations.filter(loc => loc.isServiceManaged);
    return {
      privateLocations,
      publicLocations
    };
  }
  mixParamsWithMonitors(monitors, paramsBySpace) {
    const configsBySpaces = {};
    const monitorSpaceIds = new Set();
    for (const monitor of monitors) {
      var _monitor$namespaces2;
      const spaceId = (_monitor$namespaces2 = monitor.namespaces) === null || _monitor$namespaces2 === void 0 ? void 0 : _monitor$namespaces2[0];
      if (!spaceId) {
        continue;
      }
      monitorSpaceIds.add(spaceId);
      const normalizedMonitor = (0, _utils.normalizeSecrets)(monitor).attributes;
      const {
        str: paramsString
      } = (0, _format_configs.mixParamsWithGlobalParams)(paramsBySpace[spaceId], normalizedMonitor);
      if (!configsBySpaces[spaceId]) {
        configsBySpaces[spaceId] = [];
      }
      configsBySpaces[spaceId].push((0, _format_configs.formatHeartbeatRequest)({
        spaceId,
        monitor: normalizedMonitor,
        configId: monitor.id
      }, paramsString));
    }
    return {
      configsBySpaces,
      monitorSpaceIds
    };
  }
}
exports.DeployPrivateLocationMonitors = DeployPrivateLocationMonitors;