"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processMonitors = void 0;
var _lodash = require("lodash");
var _utils = require("../../routes/overview_status/utils");
var _runtime_types = require("../../../common/runtime_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const processMonitors = (allMonitors, queryLocations) => {
  /**
   * Walk through all monitor saved objects, bucket IDs by disabled/enabled status.
   *
   * Track max period to make sure the snapshot query should reach back far enough to catch
   * latest ping for all enabled monitors.
   */
  const enabledMonitorQueryIds = [];
  const disabledMonitorQueryIds = [];
  let disabledCount = 0;
  let disabledMonitorsCount = 0;
  let projectMonitorsCount = 0;
  const allIds = [];
  let listOfLocationsSet = new Set();
  const monitorQueryIdToConfigIdMap = {};
  const monitorsData = {};
  for (const monitor of allMonitors) {
    const attrs = monitor.attributes;
    allIds.push(attrs[_runtime_types.ConfigKey.MONITOR_QUERY_ID]);
    projectMonitorsCount += (attrs === null || attrs === void 0 ? void 0 : attrs[_runtime_types.ConfigKey.MONITOR_SOURCE_TYPE]) === _runtime_types.SourceType.PROJECT ? 1 : 0;
    monitorQueryIdToConfigIdMap[attrs[_runtime_types.ConfigKey.MONITOR_QUERY_ID]] = attrs[_runtime_types.ConfigKey.CONFIG_ID];
    const monitorLocIds = attrs[_runtime_types.ConfigKey.LOCATIONS].map(location => location.id);
    if (attrs[_runtime_types.ConfigKey.ENABLED] === false) {
      const queriedLocations = Array.isArray(queryLocations) ? queryLocations : [queryLocations];
      const intersectingLocations = (0, _lodash.intersection)(monitorLocIds, queryLocations ? queriedLocations : monitorLocIds);
      disabledCount += intersectingLocations.length;
      disabledMonitorsCount += 1;
      disabledMonitorQueryIds.push(attrs[_runtime_types.ConfigKey.MONITOR_QUERY_ID]);
    } else {
      enabledMonitorQueryIds.push(attrs[_runtime_types.ConfigKey.MONITOR_QUERY_ID]);
      monitorsData[attrs[_runtime_types.ConfigKey.MONITOR_QUERY_ID]] = {
        scheduleInMs: (0, _utils.periodToMs)(attrs[_runtime_types.ConfigKey.SCHEDULE]),
        locations: queryLocations ? (0, _lodash.intersection)(monitorLocIds, queryLocations) : monitorLocIds,
        type: attrs[_runtime_types.ConfigKey.MONITOR_TYPE]
      };
      listOfLocationsSet = new Set([...listOfLocationsSet, ...monitorLocIds]);
    }
  }
  return {
    maxPeriod: Math.max(...Object.values(monitorsData).map(({
      scheduleInMs
    }) => scheduleInMs)),
    allIds,
    enabledMonitorQueryIds,
    disabledMonitorQueryIds,
    disabledCount,
    disabledMonitorsCount,
    projectMonitorsCount,
    monitorLocationIds: [...listOfLocationsSet],
    monitorQueryIdToConfigIdMap,
    monitorsData
  };
};
exports.processMonitors = processMonitors;