"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SUMMARIES_PAGE_SIZE = exports.OverviewStatusService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _moment = _interopRequireDefault(require("moment/moment"));
var _lodash = require("lodash");
var _with_apm_span = require("@kbn/apm-data-access-plugin/server/utils/with_apm_span");
var _constants = require("@kbn/security-plugin/common/constants");
var _as_mutable_array = require("../../../common/utils/as_mutable_array");
var _common = require("../common");
var _process_monitors = require("../../saved_objects/synthetics_monitor/process_monitors");
var _monitor_management = require("../../../common/constants/monitor_management");
var _alert_config = require("../../../common/runtime_types/monitor_management/alert_config");
var _client_defaults = require("../../../common/constants/client_defaults");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SUMMARIES_PAGE_SIZE = exports.SUMMARIES_PAGE_SIZE = 5000;
class OverviewStatusService {
  constructor(routeContext) {
    (0, _defineProperty2.default)(this, "filterData", {});
    this.routeContext = routeContext;
  }
  async getOverviewStatus() {
    var _this$filterData;
    this.filterData = await (0, _common.getMonitorFilters)(this.routeContext);
    const [allConfigs, statusResult] = await Promise.all([this.getMonitorConfigs(), this.getQueryResult()]);
    const {
      up,
      down,
      pending,
      upConfigs,
      downConfigs,
      pendingConfigs,
      disabledConfigs
    } = this.processOverviewStatus(allConfigs, statusResult);
    const {
      enabledMonitorQueryIds,
      disabledMonitorQueryIds,
      allIds,
      disabledCount,
      disabledMonitorsCount,
      projectMonitorsCount
    } = (0, _process_monitors.processMonitors)(allConfigs, (_this$filterData = this.filterData) === null || _this$filterData === void 0 ? void 0 : _this$filterData.locationIds);
    return {
      allIds,
      allMonitorsCount: allConfigs.length,
      disabledMonitorsCount,
      projectMonitorsCount,
      enabledMonitorQueryIds,
      disabledMonitorQueryIds,
      disabledCount,
      up,
      down,
      pending,
      upConfigs,
      downConfigs,
      pendingConfigs,
      disabledConfigs
    };
  }
  getEsDataFilters() {
    const {
      spaceId,
      request
    } = this.routeContext;
    const params = request.query || {};
    const {
      scopeStatusByLocation = true,
      tags,
      monitorTypes,
      projects,
      showFromAllSpaces
    } = params;
    const {
      locationIds
    } = this.filterData;
    const getTermFilter = (field, value) => {
      if (!value || (0, _lodash.isEmpty)(value)) {
        return [];
      }
      if (Array.isArray(value)) {
        return [{
          terms: {
            [field]: value
          }
        }];
      }
      return [{
        term: {
          [field]: value
        }
      }];
    };
    const filters = [...(showFromAllSpaces ? [] : [{
      terms: {
        'meta.space_id': [spaceId, _constants.ALL_SPACES_ID]
      }
    }]), ...getTermFilter('monitor.type', monitorTypes), ...getTermFilter('tags', tags), ...getTermFilter('monitor.project.id', projects)];
    if (scopeStatusByLocation && !(0, _lodash.isEmpty)(locationIds) && locationIds) {
      filters.push({
        terms: {
          'observer.name': locationIds
        }
      });
    }
    return filters;
  }
  async getQueryResult() {
    return (0, _with_apm_span.withApmSpan)('monitor_status_data', async () => {
      const range = {
        // max monitor schedule period is 4 hours, 20 minute subtraction is to be on safe side
        from: (0, _moment.default)().subtract(4, 'hours').subtract(20, 'minutes').toISOString(),
        to: 'now'
      };
      let hasMoreData = true;
      const monitorByIds = new Map();
      let afterKey;
      let count = 0;
      do {
        var _result$body$aggregat, _data$buckets;
        const result = await this.routeContext.syntheticsEsClient.search({
          body: {
            size: 0,
            query: {
              bool: {
                filter: [_client_defaults.FINAL_SUMMARY_FILTER, (0, _client_defaults.getRangeFilter)({
                  from: range.from,
                  to: range.to
                }), (0, _client_defaults.getTimespanFilter)({
                  from: 'now-15m',
                  to: 'now'
                }), ...this.getEsDataFilters()]
              }
            },
            aggs: {
              monitors: {
                composite: {
                  size: SUMMARIES_PAGE_SIZE,
                  sources: (0, _as_mutable_array.asMutableArray)([{
                    monitorId: {
                      terms: {
                        field: 'monitor.id'
                      }
                    }
                  }, {
                    locationId: {
                      terms: {
                        field: 'observer.name'
                      }
                    }
                  }]),
                  after: afterKey
                },
                aggs: {
                  status: {
                    top_metrics: {
                      metrics: [{
                        field: 'monitor.status'
                      }, {
                        field: 'url.full.keyword'
                      }],
                      sort: {
                        '@timestamp': 'desc'
                      }
                    }
                  }
                }
              }
            }
          }
        }, 'getCurrentStatusOverview' + count);
        count += 1;
        const data = (_result$body$aggregat = result.body.aggregations) === null || _result$body$aggregat === void 0 ? void 0 : _result$body$aggregat.monitors;
        hasMoreData = ((_data$buckets = data === null || data === void 0 ? void 0 : data.buckets) !== null && _data$buckets !== void 0 ? _data$buckets : []).length >= SUMMARIES_PAGE_SIZE;
        afterKey = data === null || data === void 0 ? void 0 : data.after_key;
        data === null || data === void 0 ? void 0 : data.buckets.forEach(({
          status: statusAgg,
          key: bKey
        }) => {
          var _statusAgg$top, _statusAgg$top$0$metr, _statusAgg$top2, _statusAgg$top2$0$met, _monitorByIds$get;
          const monitorId = String(bKey.monitorId);
          const locationId = String(bKey.locationId);
          const status = String((_statusAgg$top = statusAgg.top) === null || _statusAgg$top === void 0 ? void 0 : (_statusAgg$top$0$metr = _statusAgg$top[0].metrics) === null || _statusAgg$top$0$metr === void 0 ? void 0 : _statusAgg$top$0$metr['monitor.status']);
          const monitorUrl = (_statusAgg$top2 = statusAgg.top) === null || _statusAgg$top2 === void 0 ? void 0 : (_statusAgg$top2$0$met = _statusAgg$top2[0].metrics) === null || _statusAgg$top2$0$met === void 0 ? void 0 : _statusAgg$top2$0$met['url.full.keyword'];
          const timestamp = String(statusAgg.top[0].sort[0]);
          if (!monitorByIds.has(String(monitorId))) {
            monitorByIds.set(monitorId, []);
          }
          (_monitorByIds$get = monitorByIds.get(monitorId)) === null || _monitorByIds$get === void 0 ? void 0 : _monitorByIds$get.push({
            status,
            locationId,
            timestamp,
            monitorUrl: monitorUrl ? String(monitorUrl) : undefined
          });
        });
      } while (hasMoreData && afterKey);
      return monitorByIds;
    });
  }
  processOverviewStatus(monitors, statusData) {
    var _this$filterData2;
    let up = 0;
    let down = 0;
    const upConfigs = {};
    const downConfigs = {};
    const pendingConfigs = {};
    const disabledConfigs = {};
    const enabledMonitors = monitors.filter(monitor => monitor.attributes[_monitor_management.ConfigKey.ENABLED]);
    const disabledMonitors = monitors.filter(monitor => !monitor.attributes[_monitor_management.ConfigKey.ENABLED]);
    const queryLocIds = (_this$filterData2 = this.filterData) === null || _this$filterData2 === void 0 ? void 0 : _this$filterData2.locationIds;
    disabledMonitors.forEach(monitor => {
      var _monitor$attributes$C;
      const monitorQueryId = monitor.attributes[_monitor_management.ConfigKey.MONITOR_QUERY_ID];
      const meta = this.getMonitorMeta(monitor);
      (_monitor$attributes$C = monitor.attributes[_monitor_management.ConfigKey.LOCATIONS]) === null || _monitor$attributes$C === void 0 ? void 0 : _monitor$attributes$C.forEach(location => {
        disabledConfigs[`${meta.configId}-${location.id}`] = {
          monitorQueryId,
          status: 'disabled',
          locationId: location.id,
          locationLabel: location.label,
          ...meta
        };
      });
    });
    enabledMonitors.forEach(monitor => {
      const monitorId = monitor.attributes[_monitor_management.ConfigKey.MONITOR_QUERY_ID];
      const monitorStatus = statusData.get(monitorId);

      // discard any locations that are not in the monitorLocationsMap for the given monitor as well as those which are
      // in monitorLocationsMap but not in listOfLocations
      const monLocations = monitor.attributes[_monitor_management.ConfigKey.LOCATIONS];
      monLocations === null || monLocations === void 0 ? void 0 : monLocations.forEach(monLocation => {
        if (!(0, _lodash.isEmpty)(queryLocIds) && !(queryLocIds !== null && queryLocIds !== void 0 && queryLocIds.includes(monLocation.id))) {
          // filter out location provided via query
          return;
        }
        const locData = monitorStatus === null || monitorStatus === void 0 ? void 0 : monitorStatus.find(loc => loc.locationId === monLocation.id);
        const metaInfo = this.getMonitorMeta(monitor);
        const meta = {
          ...metaInfo,
          monitorQueryId: monitorId,
          locationId: monLocation.id,
          timestamp: locData === null || locData === void 0 ? void 0 : locData.timestamp,
          locationLabel: monLocation.label,
          urls: monitor.attributes[_monitor_management.ConfigKey.URLS] || (locData === null || locData === void 0 ? void 0 : locData.monitorUrl)
        };
        const monLocId = `${meta.configId}-${monLocation.id}`;
        if (locData) {
          if (locData.status === 'down') {
            down += 1;
            downConfigs[monLocId] = {
              ...meta,
              status: 'down'
            };
          } else if (locData.status === 'up') {
            up += 1;
            upConfigs[monLocId] = {
              ...meta,
              status: 'up'
            };
          }
        } else {
          pendingConfigs[monLocId] = {
            status: 'unknown',
            ...meta
          };
        }
      });
    });
    return {
      up,
      down,
      pending: Object.values(pendingConfigs).length,
      upConfigs,
      downConfigs,
      pendingConfigs,
      disabledConfigs
    };
  }
  async getMonitorConfigs() {
    const {
      request
    } = this.routeContext;
    const {
      query,
      showFromAllSpaces
    } = request.query || {};
    /**
     * Walk through all monitor saved objects, bucket IDs by disabled/enabled status.
     *
     * Track max period to make sure the snapshot query should reach back far enough to catch
     * latest ping for all enabled monitors.
     */

    const {
      filtersStr
    } = this.filterData;
    return this.routeContext.monitorConfigRepository.getAll({
      showFromAllSpaces,
      search: query,
      filter: filtersStr,
      fields: [_monitor_management.ConfigKey.ENABLED, _monitor_management.ConfigKey.LOCATIONS, _monitor_management.ConfigKey.MONITOR_QUERY_ID, _monitor_management.ConfigKey.CONFIG_ID, _monitor_management.ConfigKey.SCHEDULE, _monitor_management.ConfigKey.MONITOR_SOURCE_TYPE, _monitor_management.ConfigKey.MONITOR_TYPE, _monitor_management.ConfigKey.NAME, _monitor_management.ConfigKey.TAGS, _monitor_management.ConfigKey.PROJECT_ID, _monitor_management.ConfigKey.ALERT_CONFIG, _monitor_management.ConfigKey.URLS, _monitor_management.ConfigKey.MAINTENANCE_WINDOWS]
    });
  }
  getMonitorMeta(monitor) {
    var _monitor$attributes$C2;
    return {
      name: monitor.attributes[_monitor_management.ConfigKey.NAME],
      configId: monitor.attributes[_monitor_management.ConfigKey.CONFIG_ID],
      schedule: monitor.attributes[_monitor_management.ConfigKey.SCHEDULE].number,
      tags: monitor.attributes[_monitor_management.ConfigKey.TAGS],
      isEnabled: monitor.attributes[_monitor_management.ConfigKey.ENABLED],
      type: monitor.attributes[_monitor_management.ConfigKey.MONITOR_TYPE],
      projectId: monitor.attributes[_monitor_management.ConfigKey.PROJECT_ID],
      isStatusAlertEnabled: (0, _alert_config.isStatusEnabled)(monitor.attributes[_monitor_management.ConfigKey.ALERT_CONFIG]),
      updated_at: monitor.updated_at,
      spaces: monitor.namespaces,
      urls: monitor.attributes[_monitor_management.ConfigKey.URLS],
      maintenanceWindows: (_monitor$attributes$C2 = monitor.attributes[_monitor_management.ConfigKey.MAINTENANCE_WINDOWS]) === null || _monitor$attributes$C2 === void 0 ? void 0 : _monitor$attributes$C2.map(mw => mw)
    };
  }
}
exports.OverviewStatusService = OverviewStatusService;