"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.unmappedFieldsRoute = exports.schemaFieldsSimulationRoute = exports.internalSchemaRoutes = void 0;
var _std = require("@kbn/std");
var _streamsSchema = require("@kbn/streams-schema");
var _zod = require("@kbn/zod");
var _constants = require("../../../../../common/constants");
var _security_error = require("../../../../lib/streams/errors/security_error");
var _stream_crud = require("../../../../lib/streams/stream_crud");
var _create_server_route = require("../../../create_server_route");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const UNMAPPED_SAMPLE_SIZE = 500;
const unmappedFieldsRoute = exports.unmappedFieldsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /internal/streams/{name}/schema/unmapped_fields',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      scopedClusterClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    const searchBody = {
      sort: [{
        '@timestamp': {
          order: 'desc'
        }
      }],
      size: UNMAPPED_SAMPLE_SIZE
    };
    const [streamDefinition, ancestors, results] = await Promise.all([streamsClient.getStream(params.path.name), streamsClient.getAncestors(params.path.name), scopedClusterClient.asCurrentUser.search({
      index: params.path.name,
      ...searchBody
    })]);
    const sourceFields = new Set();
    results.hits.hits.forEach(hit => {
      Object.keys((0, _std.getFlattenedObject)(hit._source)).forEach(field => {
        sourceFields.add(field);
      });
    });

    // Mapped fields from the stream's definition and inherited from ancestors
    const mappedFields = new Set();
    if (_streamsSchema.Streams.WiredStream.Definition.is(streamDefinition)) {
      Object.keys(streamDefinition.ingest.wired.fields).forEach(name => mappedFields.add(name));
    }
    for (const ancestor of ancestors) {
      Object.keys(ancestor.ingest.wired.fields).forEach(name => mappedFields.add(name));
    }
    const unmappedFields = Array.from(sourceFields).filter(field => !mappedFields.has(field)).sort();
    return {
      unmappedFields
    };
  }
});
const FIELD_SIMILATION_SAMPLE_SIZE = 200;
const schemaFieldsSimulationRoute = exports.schemaFieldsSimulationRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'POST /internal/streams/{name}/schema/fields_simulation',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    body: _zod.z.object({
      field_definitions: _zod.z.array(_zod.z.intersection(_streamsSchema.fieldDefinitionConfigSchema, _zod.z.object({
        name: _zod.z.string()
      })))
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    var _sampleResults$hits$t;
    const {
      scopedClusterClient
    } = await getScopedClients({
      request
    });
    const {
      read
    } = await (0, _stream_crud.checkAccess)({
      name: params.path.name,
      scopedClusterClient
    });
    if (!read) {
      throw new _security_error.SecurityError(`Cannot read stream ${params.path.name}, insufficient privileges`);
    }
    const userFieldDefinitions = params.body.field_definitions.flatMap(field => {
      // filter out potential system fields since we can't simulate them anyway
      if (field.type === 'system') {
        return [];
      }
      return [field];
    });
    const propertiesForSample = Object.fromEntries(userFieldDefinitions.map(field => [field.name, {
      type: 'keyword'
    }]));
    const documentSamplesSearchBody = {
      // Add keyword runtime mappings so we can pair with exists, this is to attempt to "miss" less documents for the simulation.
      runtime_mappings: propertiesForSample,
      query: {
        bool: {
          filter: Object.keys(propertiesForSample).map(field => ({
            exists: {
              field
            }
          }))
        }
      },
      sort: [{
        '@timestamp': {
          order: 'desc'
        }
      }],
      size: FIELD_SIMILATION_SAMPLE_SIZE
    };
    const sampleResults = await scopedClusterClient.asCurrentUser.search({
      index: params.path.name,
      ...documentSamplesSearchBody
    });
    if (typeof sampleResults.hits.total === 'object' && ((_sampleResults$hits$t = sampleResults.hits.total) === null || _sampleResults$hits$t === void 0 ? void 0 : _sampleResults$hits$t.value) === 0 || sampleResults.hits.total === 0 || !sampleResults.hits.total) {
      return {
        status: 'unknown',
        simulationError: null,
        documentsWithRuntimeFieldsApplied: null
      };
    }
    const propertiesForSimulation = Object.fromEntries(userFieldDefinitions.map(({
      name,
      ...field
    }) => [name, field]));
    const fieldDefinitionKeys = Object.keys(propertiesForSimulation);
    const sampleResultsAsSimulationDocs = sampleResults.hits.hits.map(hit => ({
      _index: params.path.name,
      _id: hit._id,
      _source: Object.fromEntries(Object.entries((0, _std.getFlattenedObject)(hit._source)).filter(([k]) => fieldDefinitionKeys.includes(k) || k === '@timestamp'))
    }));
    const simulationBody = {
      docs: sampleResultsAsSimulationDocs,
      component_template_substitutions: {
        [`${params.path.name}@stream.layer`]: {
          template: {
            mappings: {
              dynamic: 'strict',
              properties: propertiesForSimulation
            }
          }
        }
      },
      // prevent double-processing
      pipeline_substitutions: {
        [`${params.path.name}@stream.processing`]: {
          processors: []
        }
      }
    };

    // TODO: We should be using scopedClusterClient.asCurrentUser.simulate.ingest() but the ES JS lib currently has a bug. The types also aren't available yet, so we use any.
    const simulation = await scopedClusterClient.asCurrentUser.transport.request({
      method: 'POST',
      path: `_ingest/_simulate`,
      body: simulationBody
    });
    const hasErrors = simulation.docs.some(doc => doc.doc.error !== undefined);
    if (hasErrors) {
      const documentWithError = simulation.docs.find(doc => {
        return doc.doc.error !== undefined;
      });
      return {
        status: 'failure',
        simulationError: JSON.stringify(
        // Use the first error as a representative error
        documentWithError.doc.error),
        documentsWithRuntimeFieldsApplied: null
      };
    }

    // Convert the field definitions to a format that can be used in runtime mappings (match_only_text -> keyword)
    const propertiesCompatibleWithRuntimeMappings = Object.fromEntries(userFieldDefinitions.map(field => [field.name, {
      type: field.type === 'match_only_text' ? 'keyword' : field.type,
      ...(field.format ? {
        format: field.format
      } : {})
    }]));
    const runtimeFieldsSearchBody = {
      runtime_mappings: propertiesCompatibleWithRuntimeMappings,
      sort: [{
        '@timestamp': {
          order: 'desc'
        }
      }],
      size: FIELD_SIMILATION_SAMPLE_SIZE,
      fields: params.body.field_definitions.map(field => field.name),
      _source: false
    };

    // This gives us a "fields" representation rather than _source from the simulation
    const runtimeFieldsResult = await scopedClusterClient.asCurrentUser.search({
      index: params.path.name,
      ...runtimeFieldsSearchBody
    });
    return {
      status: 'success',
      simulationError: null,
      documentsWithRuntimeFieldsApplied: runtimeFieldsResult.hits.hits.map(hit => {
        if (!hit.fields) {
          return {};
        }
        return Object.keys(hit.fields).reduce((acc, field) => {
          acc[field] = hit.fields[field][0];
          return acc;
        }, {});
      }).filter(doc => Object.keys(doc).length > 0)
    };
  }
});
const internalSchemaRoutes = exports.internalSchemaRoutes = {
  ...unmappedFieldsRoute,
  ...schemaFieldsSimulationRoute
};