"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.unmanagedAssetDetailsRoute = exports.internalManagementRoutes = void 0;
var _streamsSchema = require("@kbn/streams-schema");
var _zod = require("@kbn/zod");
var _constants = require("../../../../../common/constants");
var _security_error = require("../../../../lib/streams/errors/security_error");
var _wrong_stream_type_error = require("../../../../lib/streams/errors/wrong_stream_type_error");
var _stream_crud = require("../../../../lib/streams/stream_crud");
var _create_server_route = require("../../../create_server_route");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const unmanagedAssetDetailsRoute = exports.unmanagedAssetDetailsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /internal/streams/{name}/_unmanaged_assets',
  options: {
    access: 'internal'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients
  }) => {
    const {
      scopedClusterClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    const {
      read
    } = await (0, _stream_crud.checkAccess)({
      name: params.path.name,
      scopedClusterClient
    });
    if (!read) {
      throw new _security_error.SecurityError(`Cannot read stream ${params.path.name}, insufficient privileges`);
    }
    const stream = await streamsClient.getStream(params.path.name);
    if (!_streamsSchema.Streams.UnwiredStream.Definition.is(stream)) {
      throw new _wrong_stream_type_error.WrongStreamTypeError(`Stream definition for ${params.path.name} is not an unwired stream`);
    }
    const dataStream = await streamsClient.getDataStream(params.path.name);
    const assets = await (0, _stream_crud.getUnmanagedElasticsearchAssets)({
      dataStream,
      scopedClusterClient
    });
    return (0, _stream_crud.getUnmanagedElasticsearchAssetDetails)({
      assets,
      scopedClusterClient
    });
  }
});
const internalManagementRoutes = exports.internalManagementRoutes = {
  ...unmanagedAssetDetailsRoute
};