"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useAsyncSample = void 0;
var _react = require("react");
var _streamsSchema = require("@kbn/streams-schema");
var _useToggle = _interopRequireDefault(require("react-use/lib/useToggle"));
var _rxjs = require("rxjs");
var _common = require("@kbn/data-plugin/common");
var _reactHooks = require("@kbn/react-hooks");
var _lodash = require("lodash");
var _use_kibana = require("../use_kibana");
var _condition = require("../../util/condition");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useAsyncSample = options => {
  const {
    dependencies: {
      start: {
        data
      }
    }
  } = (0, _use_kibana.useKibana)();
  const controller = (0, _reactHooks.useAbortController)();

  // Documents
  const [isLoadingDocuments, toggleIsLoadingDocuments] = (0, _useToggle.default)(false);
  const [documentsError, setDocumentsError] = (0, _react.useState)();
  const [documents, setDocuments] = (0, _react.useState)([]);

  // Document counts / percentage
  const [isLoadingDocumentCounts, toggleIsLoadingDocumentCounts] = (0, _useToggle.default)(false);
  const [documentCountsError, setDocumentCountsError] = (0, _react.useState)();
  const [approximateMatchingPercentage, setApproximateMatchingPercentage] = (0, _react.useState)();
  const [refreshId, setRefreshId] = (0, _react.useState)(0);
  const convertedCondition = (0, _react.useMemo)(() => {
    const condition = options.condition ? (0, _condition.emptyEqualsToAlways)(options.condition) : undefined;
    return condition && (0, _streamsSchema.isAlwaysCondition)(condition) ? undefined : condition;
  }, [options.condition]);
  const refresh = (0, _react.useCallback)(() => {
    return setRefreshId(id => id + 1);
  }, []);
  (0, _react.useEffect)(() => {
    if (!options.start || !options.end) {
      setDocuments([]);
      setApproximateMatchingPercentage(undefined);
      return;
    }
    const runtimeMappings = getRuntimeMappings(options.streamDefinition, convertedCondition);

    // Documents
    toggleIsLoadingDocuments(true);
    const documentSubscription = data.search.search({
      params: {
        index: options.streamDefinition.stream.name,
        body: getDocumentsSearchBody(options, runtimeMappings, convertedCondition)
      }
    }, {
      abortSignal: controller.signal
    }).subscribe({
      next: result => {
        var _result$rawResponse$h2;
        if (!(0, _common.isRunningResponse)(result)) {
          var _result$rawResponse$h;
          toggleIsLoadingDocuments(false);
          if ((_result$rawResponse$h = result.rawResponse.hits) !== null && _result$rawResponse$h !== void 0 && _result$rawResponse$h.hits) {
            setDocuments(result.rawResponse.hits.hits.map(hit => hit._source));
          }
          return;
        } else if (!(0, _lodash.isEmpty)((_result$rawResponse$h2 = result.rawResponse.hits) === null || _result$rawResponse$h2 === void 0 ? void 0 : _result$rawResponse$h2.hits)) {
          setDocuments(result.rawResponse.hits.hits.map(hit => hit._source));
        }
      },
      error: e => {
        setDocumentsError(e);
        toggleIsLoadingDocuments(false);
      }
    });
    toggleIsLoadingDocumentCounts(true);
    setApproximateMatchingPercentage(undefined);
    const documentCountsSubscription = data.search.search({
      params: {
        index: options.streamDefinition.stream.name,
        body: getDocumentCountForSampleRateSearchBody(options)
      }
    }, {
      abortSignal: controller.signal
    }).pipe((0, _rxjs.filter)(result => !(0, _common.isRunningResponse)(result)), (0, _rxjs.switchMap)(response => {
      const docCount = response.rawResponse.hits.total && typeof response.rawResponse.hits.total !== 'number' && 'value' in response.rawResponse.hits.total ? response.rawResponse.hits.total.value : response.rawResponse.hits.total;
      const probability = calculateProbability(docCount);
      return data.search.search({
        params: {
          index: options.streamDefinition.stream.name,
          body: getDocumentCountsSearchBody(options, runtimeMappings, probability, convertedCondition)
        }
      }, {
        abortSignal: controller.signal
      });
    })).subscribe({
      next: result => {
        var _result$rawResponse;
        if (!(0, _common.isRunningResponse)(result)) {
          toggleIsLoadingDocumentCounts(false);
        }
        // Aggregations don't return partial results so we just wait until the end
        if ((_result$rawResponse = result.rawResponse) !== null && _result$rawResponse !== void 0 && _result$rawResponse.aggregations) {
          // We need to divide this by the sampling / probability factor:
          // https://www.elastic.co/guide/en/elasticsearch/reference/current/search-aggregations-random-sampler-aggregation.html#random-sampler-special-cases
          const sampleAgg = result.rawResponse.aggregations.sample;
          const randomSampleDocCount = sampleAgg.doc_count / sampleAgg.probability;
          const matchingDocCount = sampleAgg.matching_docs.doc_count;
          const percentage = 100 * matchingDocCount / randomSampleDocCount;
          setApproximateMatchingPercentage(percentage.toFixed(2));
        }
      },
      error: e => {
        toggleIsLoadingDocumentCounts(false);
        setDocumentCountsError(e);
      }
    });
    return () => {
      documentSubscription.unsubscribe();
      documentCountsSubscription.unsubscribe();
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [data.search, convertedCondition, options.start, options.end, options.size, options.streamDefinition, refreshId]);
  return {
    isLoadingDocuments,
    documentsError,
    documents,
    isLoadingDocumentCounts,
    documentCountsError,
    approximateMatchingPercentage,
    refresh
  };
};
exports.useAsyncSample = useAsyncSample;
// Create runtime mappings for fields that aren't mapped.
// Conditions could be using fields which are not indexed or they could use it with other types than they are eventually mapped as.
// Because of this we can't rely on mapped fields to draw a sample, instead we need to use runtime fields to simulate what happens during
// ingest in the painless condition checks.
const getRuntimeMappings = (streamDefinition, condition) => {
  if (!condition) return {};
  const wiredMappedFields = 'wired' in streamDefinition.stream.ingest ? streamDefinition.stream.ingest.wired.fields : {};
  const mappedFields = Object.keys(wiredMappedFields).concat(Object.keys(streamDefinition.inherited_fields));
  return Object.fromEntries((0, _streamsSchema.getConditionFields)(condition).filter(field => !mappedFields.includes(field.name)).map(field => [field.name, {
    type: field.type === 'string' ? 'keyword' : 'double'
  }]));
};
const getDocumentsSearchBody = (options, runtimeMappings, condition) => {
  const {
    size,
    start,
    end
  } = options;
  const searchBody = {
    query: {
      bool: {
        must: [condition ? (0, _streamsSchema.conditionToQueryDsl)(condition) : {
          match_all: {}
        }, {
          range: {
            '@timestamp': {
              gte: start,
              lte: end,
              format: 'epoch_millis'
            }
          }
        }]
      }
    },
    runtime_mappings: runtimeMappings,
    sort: [{
      '@timestamp': {
        order: 'desc'
      }
    }],
    terminate_after: size,
    track_total_hits: false,
    size
  };
  return searchBody;
};
const getDocumentCountForSampleRateSearchBody = options => {
  const {
    start,
    end
  } = options;
  const searchBody = {
    query: {
      range: {
        '@timestamp': {
          gte: start,
          lte: end,
          format: 'epoch_millis'
        }
      }
    },
    track_total_hits: true,
    size: 0
  };
  return searchBody;
};
const getDocumentCountsSearchBody = (options, runtimeMappings, probability, condition) => {
  const {
    start,
    end
  } = options;
  const searchBody = {
    query: {
      bool: {
        must: [{
          range: {
            '@timestamp': {
              gte: start,
              lte: end,
              format: 'epoch_millis'
            }
          }
        }]
      }
    },
    aggs: {
      sample: {
        random_sampler: {
          probability
        },
        aggs: {
          matching_docs: {
            filter: condition ? (0, _streamsSchema.conditionToQueryDsl)(condition) : {
              match_all: {}
            }
          }
        }
      }
    },
    runtime_mappings: runtimeMappings,
    size: 0,
    _source: false,
    track_total_hits: false
  };
  return searchBody;
};
const calculateProbability = docCount => {
  if (!docCount) return 1;
  const probabilityThreshold = 100000;
  if (docCount > probabilityThreshold) {
    const probability = probabilityThreshold / docCount;
    // Values between 0.5 and 1 are not supported by the random sampler
    return probability <= 0.5 ? probability : 1;
  } else {
    return 1;
  }
};