"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createExternalService = exports.SYS_DICTIONARY_ENDPOINT = void 0;
var _axios_utils = require("@kbn/actions-plugin/server/lib/axios_utils");
var _lodash = require("lodash");
var i18n = _interopRequireWildcard(require("./translations"));
var _utils = require("./utils");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SYS_DICTIONARY_ENDPOINT = exports.SYS_DICTIONARY_ENDPOINT = `api/now/table/sys_dictionary`;
const createExternalService = ({
  credentials,
  logger,
  configurationUtilities,
  serviceConfig,
  axiosInstance,
  connectorUsageCollector
}) => {
  const {
    config,
    secrets
  } = credentials;
  const {
    table,
    importSetTable,
    useImportAPI,
    appScope
  } = serviceConfig;
  const {
    apiUrl: url,
    usesTableApi: usesTableApiConfigValue,
    isOAuth,
    clientId,
    jwtKeyId,
    userIdentifierValue
  } = config;
  const {
    username,
    password,
    clientSecret,
    privateKey
  } = secrets;
  if (!url || !isOAuth && (!username || !password) || isOAuth && (!clientSecret || !privateKey || !clientId || !jwtKeyId || !userIdentifierValue)) {
    throw Error(`[Action]${i18n.SERVICENOW}: Wrong configuration.`);
  }
  const urlWithoutTrailingSlash = url.endsWith('/') ? url.slice(0, -1) : url;
  const importSetTableUrl = `${urlWithoutTrailingSlash}/api/now/import/${importSetTable}`;
  const tableApiIncidentUrl = `${urlWithoutTrailingSlash}/api/now/v2/table/${table}`;
  const fieldsUrl = `${urlWithoutTrailingSlash}/${SYS_DICTIONARY_ENDPOINT}?sysparm_query=name=task^ORname=${table}^internal_type=string&active=true&array=false&read_only=false&sysparm_fields=max_length,element,column_label,mandatory`;
  const choicesUrl = `${urlWithoutTrailingSlash}/api/now/table/sys_choice`;
  /**
   * Need to be set the same at:
   * x-pack/platform/plugins/shared/triggers_actions_ui/public/application/components/builtin_action_types/servicenow/api.ts
   */
  const getVersionUrl = () => `${urlWithoutTrailingSlash}/api/${appScope}/elastic_api/health`;
  const useTableApi = !useImportAPI || usesTableApiConfigValue;
  const getCreateIncidentUrl = () => useTableApi ? tableApiIncidentUrl : importSetTableUrl;
  const getUpdateIncidentUrl = incidentId => useTableApi ? `${tableApiIncidentUrl}/${incidentId}` : importSetTableUrl;
  const getIncidentViewURL = id => {
    // Based on: https://docs.servicenow.com/bundle/orlando-platform-user-interface/page/use/navigation/reference/r_NavigatingByURLExamples.html
    return `${urlWithoutTrailingSlash}/nav_to.do?uri=${table}.do?sys_id=${id}`;
  };
  const getIncidentByCorrelationIdUrl = correlationId => {
    return `${tableApiIncidentUrl}?sysparm_query=ORDERBYDESCsys_created_on^correlation_id=${correlationId}`;
  };
  const getChoicesURL = fields => {
    const elements = fields.slice(1).reduce((acc, field) => `${acc}^ORelement=${field}`, `element=${fields[0]}`);
    return `${choicesUrl}?sysparm_query=name=task^ORname=${table}^${elements}^language=en&sysparm_fields=label,value,dependent_value,element`;
  };
  const checkInstance = res => {
    if (res.status >= 200 && res.status < 400 && res.data.result == null) {
      var _res$request$connecti, _res$request, _res$request$connecti2;
      throw new Error(`There is an issue with your Service Now Instance. Please check ${(_res$request$connecti = (_res$request = res.request) === null || _res$request === void 0 ? void 0 : (_res$request$connecti2 = _res$request.connection) === null || _res$request$connecti2 === void 0 ? void 0 : _res$request$connecti2.servername) !== null && _res$request$connecti !== void 0 ? _res$request$connecti : ''}.`);
    }
  };
  const isImportSetApiResponseAnError = data => data.status === 'error';
  const throwIfImportSetApiResponseIsAnError = res => {
    if (res.result.length === 0) {
      throw new Error('Unexpected result');
    }
    const data = res.result[0];

    // Create ResponseError message?
    if (isImportSetApiResponseAnError(data)) {
      throw new Error(data.error_message);
    }
  };

  /**
   * Gets the Elastic SN Application information including the current version.
   * It should not be used on connectors that use the old API.
   */
  const getApplicationInformation = async () => {
    try {
      const res = await (0, _axios_utils.request)({
        axios: axiosInstance,
        url: getVersionUrl(),
        logger,
        configurationUtilities,
        method: 'get',
        connectorUsageCollector // TODO check if this is internal
      });
      checkInstance(res);
      return {
        ...res.data.result
      };
    } catch (error) {
      throw (0, _utils.createServiceError)(error, 'Unable to get application version');
    }
  };
  const logApplicationInfo = (scope, version) => logger.debug(`Create incident: Application scope: ${scope}: Application version${version}`);
  const checkIfApplicationIsInstalled = async () => {
    if (!useTableApi) {
      const {
        version,
        scope
      } = await getApplicationInformation();
      logApplicationInfo(scope, version);
    }
  };
  const getIncident = async id => {
    try {
      if ((id === null || id === void 0 ? void 0 : id.trim()) === '') {
        throw new Error('Incident id is empty.');
      }
      const res = await (0, _axios_utils.request)({
        axios: axiosInstance,
        url: `${tableApiIncidentUrl}/${id}`,
        logger,
        configurationUtilities,
        method: 'get',
        connectorUsageCollector
      });
      checkInstance(res);
      return {
        ...res.data.result
      };
    } catch (error) {
      throw (0, _utils.createServiceError)(error, `Unable to get incident with id ${id}`);
    }
  };
  const findIncidents = async params => {
    try {
      const res = await (0, _axios_utils.request)({
        axios: axiosInstance,
        url: tableApiIncidentUrl,
        logger,
        params,
        configurationUtilities,
        connectorUsageCollector
      });
      checkInstance(res);
      return res.data.result.length > 0 ? {
        ...res.data.result
      } : undefined;
    } catch (error) {
      throw (0, _utils.createServiceError)(error, 'Unable to find incidents by query');
    }
  };
  const getUrl = () => urlWithoutTrailingSlash;
  const createIncident = async ({
    incident
  }) => {
    try {
      (0, _utils.throwIfAdditionalFieldsNotSupported)(useTableApi, incident);
      await checkIfApplicationIsInstalled();
      const res = await (0, _axios_utils.request)({
        axios: axiosInstance,
        url: getCreateIncidentUrl(),
        logger,
        method: 'post',
        data: (0, _utils.prepareIncident)(useTableApi, incident),
        configurationUtilities,
        connectorUsageCollector
      });
      checkInstance(res);
      if (!useTableApi) {
        throwIfImportSetApiResponseIsAnError(res.data);
      }
      const incidentId = useTableApi ? res.data.result.sys_id : res.data.result[0].sys_id;
      const insertedIncident = await getIncident(incidentId);
      return {
        title: insertedIncident.number,
        id: insertedIncident.sys_id,
        pushedDate: (0, _utils.getPushedDate)(insertedIncident.sys_created_on),
        url: getIncidentViewURL(insertedIncident.sys_id)
      };
    } catch (error) {
      throw (0, _utils.createServiceError)(error, 'Unable to create incident');
    }
  };
  const updateIncident = async ({
    incidentId,
    incident
  }) => {
    try {
      (0, _utils.throwIfAdditionalFieldsNotSupported)(useTableApi, incident);
      await checkIfApplicationIsInstalled();
      const res = await (0, _axios_utils.request)({
        axios: axiosInstance,
        url: getUpdateIncidentUrl(incidentId),
        // Import Set API supports only POST.
        method: useTableApi ? 'patch' : 'post',
        logger,
        data: {
          ...(0, _utils.prepareIncident)(useTableApi, incident),
          // elastic_incident_id is used to update the incident when using the Import Set API.
          ...(useTableApi ? {} : {
            elastic_incident_id: incidentId
          })
        },
        configurationUtilities,
        connectorUsageCollector
      });
      checkInstance(res);
      if (!useTableApi) {
        throwIfImportSetApiResponseIsAnError(res.data);
      }
      const id = useTableApi ? res.data.result.sys_id : res.data.result[0].sys_id;
      const updatedIncident = await getIncident(id);
      return {
        title: updatedIncident.number,
        id: updatedIncident.sys_id,
        pushedDate: (0, _utils.getPushedDate)(updatedIncident.sys_updated_on),
        url: getIncidentViewURL(updatedIncident.sys_id)
      };
    } catch (error) {
      throw (0, _utils.createServiceError)(error, `Unable to update incident with id ${incidentId}`);
    }
  };
  const getIncidentByCorrelationId = async correlationId => {
    try {
      var _res$data$result$;
      if ((correlationId === null || correlationId === void 0 ? void 0 : correlationId.trim()) === '') {
        throw new Error('Correlation ID is empty.');
      }
      const res = await (0, _axios_utils.request)({
        axios: axiosInstance,
        url: getIncidentByCorrelationIdUrl(correlationId),
        method: 'get',
        logger,
        configurationUtilities,
        connectorUsageCollector
      });
      checkInstance(res);
      const foundIncident = (_res$data$result$ = res.data.result[0]) !== null && _res$data$result$ !== void 0 ? _res$data$result$ : null;
      return foundIncident;
    } catch (error) {
      throw (0, _utils.createServiceError)(error, `Unable to get incident by correlation ID ${correlationId}`);
    }
  };
  const closeIncident = async params => {
    try {
      const {
        correlationId,
        incidentId
      } = params;
      let incidentToBeClosed = null;
      if (correlationId == null && incidentId == null) {
        throw new Error('No correlationId or incidentId found.');
      }
      if (incidentId) {
        incidentToBeClosed = await getIncident(incidentId);
      } else if (correlationId) {
        incidentToBeClosed = await getIncidentByCorrelationId(correlationId);
      }
      if (incidentToBeClosed === null || (0, _lodash.isEmpty)(incidentToBeClosed)) {
        logger.warn(`[ServiceNow][CloseIncident] No incident found with correlation_id: ${correlationId} or incidentId: ${incidentId}.`);
        return null;
      }
      if (incidentToBeClosed.state === '7') {
        logger.warn(`[ServiceNow][CloseIncident] Incident with correlation_id: ${correlationId} or incidentId: ${incidentId} is closed.`);
        return {
          title: incidentToBeClosed.number,
          id: incidentToBeClosed.sys_id,
          pushedDate: (0, _utils.getPushedDate)(incidentToBeClosed.sys_updated_on),
          url: getIncidentViewURL(incidentToBeClosed.sys_id)
        };
      }
      const closedIncident = await updateIncident({
        incidentId: incidentToBeClosed.sys_id,
        incident: {
          state: '7',
          // used for "closed" status in serviceNow
          close_code: 'Closed/Resolved by Caller',
          close_notes: 'Closed by Caller'
        }
      });
      return closedIncident;
    } catch (error) {
      var _error$response;
      if ((error === null || error === void 0 ? void 0 : (_error$response = error.response) === null || _error$response === void 0 ? void 0 : _error$response.status) === 404) {
        logger.warn(`[ServiceNow][CloseIncident] No incident found with incidentId: ${params.incidentId}.`);
        return null;
      }
      throw (0, _utils.createServiceError)(error, 'Unable to close incident');
    }
  };
  const getFields = async () => {
    try {
      const res = await (0, _axios_utils.request)({
        axios: axiosInstance,
        url: fieldsUrl,
        logger,
        configurationUtilities,
        connectorUsageCollector
      });
      checkInstance(res);
      return res.data.result.length > 0 ? res.data.result : [];
    } catch (error) {
      throw (0, _utils.createServiceError)(error, 'Unable to get fields');
    }
  };
  const getChoices = async fields => {
    try {
      const res = await (0, _axios_utils.request)({
        axios: axiosInstance,
        url: getChoicesURL(fields),
        logger,
        configurationUtilities,
        connectorUsageCollector
      });
      checkInstance(res);
      return res.data.result;
    } catch (error) {
      throw (0, _utils.createServiceError)(error, 'Unable to get choices');
    }
  };
  return {
    createIncident,
    findIncidents,
    getFields,
    getIncident,
    updateIncident,
    getChoices,
    getUrl,
    checkInstance,
    getApplicationInformation,
    checkIfApplicationIsInstalled,
    closeIncident,
    getIncidentByCorrelationId
  };
};
exports.createExternalService = createExternalService;