"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DefaultTransformManager = void 0;
var _errors = require("../errors");
var _retry = require("../utils/retry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DefaultTransformManager {
  constructor(generators, scopedClusterClient, logger, abortController = new AbortController()) {
    this.generators = generators;
    this.scopedClusterClient = scopedClusterClient;
    this.logger = logger;
    this.abortController = abortController;
  }
  async install(slo) {
    const generator = this.generators[slo.indicator.type];
    if (!generator) {
      this.logger.debug(`No transform generator found for indicator type [${slo.indicator.type}]`);
      throw new Error(`Unsupported indicator type [${slo.indicator.type}]`);
    }
    const transformParams = await generator.getTransformParams(slo);
    try {
      await (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.transform.putTransform(transformParams, {
        signal: this.abortController.signal
      }), {
        logger: this.logger
      });
    } catch (err) {
      var _err$meta, _err$meta$body, _err$meta$body$error;
      this.logger.debug(`Cannot create SLO transform for indicator type [${slo.indicator.type}]. ${err}`);
      if (((_err$meta = err.meta) === null || _err$meta === void 0 ? void 0 : (_err$meta$body = _err$meta.body) === null || _err$meta$body === void 0 ? void 0 : (_err$meta$body$error = _err$meta$body.error) === null || _err$meta$body$error === void 0 ? void 0 : _err$meta$body$error.type) === 'security_exception') {
        throw new _errors.SecurityException(err.meta.body.error.reason);
      }
      throw err;
    }
    return transformParams.transform_id;
  }
  async inspect(slo) {
    const generator = this.generators[slo.indicator.type];
    if (!generator) {
      this.logger.debug(`No transform generator found for indicator type [${slo.indicator.type}]`);
      throw new Error(`Unsupported indicator type [${slo.indicator.type}]`);
    }
    return await generator.getTransformParams(slo);
  }
  async preview(transformId) {
    try {
      await (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.transform.previewTransform({
        transform_id: transformId
      }, {
        signal: this.abortController.signal
      }), {
        logger: this.logger
      });
    } catch (err) {
      this.logger.debug(`Cannot preview SLO transform [${transformId}]. ${err}`);
      throw err;
    }
  }
  async start(transformId) {
    try {
      await (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.transform.startTransform({
        transform_id: transformId
      }, {
        ignore: [409],
        signal: this.abortController.signal
      }), {
        logger: this.logger
      });
      await this.scheduleNowTransform(transformId);
    } catch (err) {
      this.logger.debug(`Cannot start SLO transform [${transformId}]. ${err}`);
      throw err;
    }
  }
  async stop(transformId) {
    try {
      await (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.transform.stopTransform({
        transform_id: transformId,
        wait_for_completion: true,
        force: true,
        allow_no_match: true
      }, {
        ignore: [404],
        signal: this.abortController.signal
      }), {
        logger: this.logger
      });
    } catch (err) {
      this.logger.debug(`Cannot stop SLO transform [${transformId}]. ${err}`);
      throw err;
    }
  }
  async uninstall(transformId) {
    try {
      await (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.transform.deleteTransform({
        transform_id: transformId,
        force: true
      }, {
        ignore: [404],
        signal: this.abortController.signal
      }), {
        logger: this.logger
      });
    } catch (err) {
      this.logger.debug(`Cannot delete SLO transform [${transformId}]. ${err}`);
      throw err;
    }
  }
  async getVersion(transformId) {
    try {
      var _response$transforms$;
      const response = await (0, _retry.retryTransientEsErrors)(() => this.scopedClusterClient.asSecondaryAuthUser.transform.getTransform({
        transform_id: transformId
      }, {
        signal: this.abortController.signal
      }), {
        logger: this.logger
      });
      if (response.count === 0) {
        return undefined;
      }
      return (_response$transforms$ = response.transforms[0]._meta) === null || _response$transforms$ === void 0 ? void 0 : _response$transforms$.version;
    } catch (err) {
      var _err$meta2, _err$meta2$body, _err$meta2$body$error;
      if (((_err$meta2 = err.meta) === null || _err$meta2 === void 0 ? void 0 : (_err$meta2$body = _err$meta2.body) === null || _err$meta2$body === void 0 ? void 0 : (_err$meta2$body$error = _err$meta2$body.error) === null || _err$meta2$body$error === void 0 ? void 0 : _err$meta2$body$error.type) === 'resource_not_found_exception') {
        return undefined;
      }
      this.logger.debug(`Cannot retrieve SLO transform version [${transformId}]. ${err}`);
      throw err;
    }
  }
  async scheduleNowTransform(transformId) {
    this.scopedClusterClient.asSecondaryAuthUser.transform.scheduleNowTransform({
      transform_id: transformId
    }, {
      signal: this.abortController.signal
    }).then(() => {
      this.logger.debug(`SLO transform [${transformId}] scheduled now successfully`);
    }).catch(e => {
      this.logger.debug(`Cannot schedule now SLO transform [${transformId}]. ${e}`);
    });
  }
}
exports.DefaultTransformManager = DefaultTransformManager;