"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformError = void 0;
var _zodHelpers = require("@kbn/zod-helpers");
var _zod = require("@kbn/zod");
var _bad_request_error = require("../bad_request_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// We can't import `isBoom` from @hapi/boom today because we get transpilation errors in Webpack 4
// due to the usage of the operator ?? inside the `@hapi/boom` library and its dependencies.
// TODO: Might be able to use the library's `isBoom` when Webpack 5 is merged (https://github.com/elastic/kibana/pull/191106)
function isBoom(err) {
  return err instanceof Error && `isBoom` in err && !!err.isBoom;
}
const transformError = err => {
  if (isBoom(err)) {
    return {
      message: err.output.payload.message,
      statusCode: err.output.statusCode
    };
  } else if (err instanceof _zod.ZodError) {
    const message = (0, _zodHelpers.stringifyZodError)(err);
    return {
      message,
      // These errors can occur when handling requests after validation and can
      // indicate of issues in business logic, so they are 500s instead of 400s
      statusCode: 500
    };
  } else {
    if (err.statusCode != null) {
      if (err.body != null && err.body.error != null) {
        return {
          statusCode: err.statusCode,
          message: `${err.body.error.type}: ${err.body.error.reason}`
        };
      } else {
        return {
          statusCode: err.statusCode,
          message: err.message
        };
      }
    } else if (err instanceof _bad_request_error.BadRequestError) {
      // allows us to throw request validation errors in the absence of Boom
      return {
        message: err.message,
        statusCode: 400
      };
    } else {
      // natively return the err and allow the regular framework
      // to deal with the error when it is a non Boom
      return {
        message: err.message != null ? err.message : '(unknown error message)',
        statusCode: 500
      };
    }
  }
};
exports.transformError = transformError;