"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createTelemetryIndicesMetadataTaskConfig = createTelemetryIndicesMetadataTaskConfig;
var _helpers = require("../helpers");
var _events = require("../event_based/events");
var _configuration = require("../configuration");
var _types = require("../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const COUNTER_LABELS = ['security_solution', 'indices-metadata'];
function createTelemetryIndicesMetadataTaskConfig() {
  const taskType = 'security:indices-metadata-telemetry';
  return {
    type: taskType,
    title: 'Security Solution Telemetry Indices Metadata task',
    interval: '24h',
    timeout: '5m',
    version: '1.0.0',
    getLastExecutionTime: _helpers.getPreviousDailyTaskTimestamp,
    runTask: async (taskId, logger, receiver, sender, taskMetricsService, taskExecutionPeriod) => {
      const mdc = {
        task_id: taskId,
        task_execution_period: taskExecutionPeriod
      };
      const log = (0, _helpers.newTelemetryLogger)(logger.get('indices-metadata'), mdc);
      const trace = taskMetricsService.start(taskType);
      const taskConfig = _configuration.telemetryConfiguration.indices_metadata_config;
      const publishDatastreamsStats = stats => {
        const events = {
          items: stats
        };
        sender.reportEBT(_events.TELEMETRY_DATA_STREAM_EVENT, events);
        log.info(`Sent data streams`, {
          count: events.items.length
        });
        return events.items.length;
      };
      const publishIndicesStats = async indices => {
        const indicesStats = {
          items: []
        };
        for await (const stat of receiver.getIndicesStats(indices, taskConfig.index_query_size)) {
          indicesStats.items.push(stat);
        }
        sender.reportEBT(_events.TELEMETRY_INDEX_STATS_EVENT, indicesStats);
        log.info(`Sent indices stats`, {
          count: indicesStats.items.length
        });
        return indicesStats.items.length;
      };
      const publishIndexTemplatesStats = async indexTemplates => {
        const templateStats = {
          items: indexTemplates
        };
        sender.reportEBT(_events.TELEMETRY_INDEX_TEMPLATES_EVENT, templateStats);
        log.info(`Sent index templates`, {
          count: indexTemplates.length
        });
        return templateStats.items.length;
      };
      const publishIndicesSettings = settings => {
        const indicesSettings = {
          items: settings
        };
        sender.reportEBT(_events.TELEMETRY_INDEX_SETTINGS_EVENT, indicesSettings);
        log.info(`Sent indices settings`, {
          count: indicesSettings.items.length
        });
        return indicesSettings.items.length;
      };
      const publishIlmStats = async indices => {
        const ilmNames = new Set();
        const ilmsStats = {
          items: []
        };
        for await (const stat of receiver.getIlmsStats(indices, taskConfig.ilm_stats_query_size)) {
          if (stat.policy_name !== undefined) {
            ilmNames.add(stat.policy_name);
            ilmsStats.items.push(stat);
          }
        }
        sender.reportEBT(_events.TELEMETRY_ILM_STATS_EVENT, ilmsStats);
        log.info(`Sent ILM stats`, {
          count: ilmNames.size
        });
        return ilmNames;
      };
      const publishIlmPolicies = async ilmNames => {
        const ilmPolicies = {
          items: []
        };
        for await (const policy of receiver.getIlmsPolicies(Array.from(ilmNames.values()), taskConfig.ilm_policy_query_size)) {
          ilmPolicies.items.push(policy);
        }
        sender.reportEBT(_events.TELEMETRY_ILM_POLICY_EVENT, ilmPolicies);
        log.info('Sent ILM policies', {
          count: ilmPolicies.items.length
        });
        return ilmPolicies.items.length;
      };
      const incrementCounter = (type, name, value) => {
        const telemetryUsageCounter = sender.getTelemetryUsageCluster();
        telemetryUsageCounter === null || telemetryUsageCounter === void 0 ? void 0 : telemetryUsageCounter.incrementCounter({
          counterName: (0, _helpers.createUsageCounterLabel)(COUNTER_LABELS.concat(name)),
          counterType: type,
          incrementBy: value
        });
      };
      try {
        // 1. Get cluster stats and list of indices and datastreams
        const [indicesSettings, dataStreams, indexTemplates] = await Promise.all([receiver.getIndices(), receiver.getDataStreams(), receiver.getIndexTemplatesStats()]);
        const indices = indicesSettings.map(index => index.index_name);

        // 2. Publish datastreams stats
        const dsCount = publishDatastreamsStats(dataStreams.slice(0, taskConfig.datastreams_threshold));
        incrementCounter(_types.TelemetryCounter.DOCS_SENT, 'datastreams-stats', dsCount);

        // 3. Publish indices settings
        const indicesSettingsCount = publishIndicesSettings(indicesSettings.slice(0, taskConfig.indices_settings_threshold));
        incrementCounter(_types.TelemetryCounter.DOCS_SENT, 'indices-settings', indicesSettingsCount);

        // 4. Get and publish indices stats
        const indicesCount = await publishIndicesStats(indices.slice(0, taskConfig.indices_threshold)).then(count => {
          incrementCounter(_types.TelemetryCounter.DOCS_SENT, 'indices-stats', count);
          return count;
        }).catch(error => {
          log.warn(`Error getting indices stats`, (0, _helpers.withErrorMessage)(error));
          incrementCounter(_types.TelemetryCounter.RUNTIME_ERROR, 'indices-stats', 1);
          return 0;
        });

        // 5. Get ILM stats and publish them
        const ilmNames = await publishIlmStats(indices.slice(0, taskConfig.indices_threshold)).then(names => {
          incrementCounter(_types.TelemetryCounter.DOCS_SENT, 'ilm-stats', names.size);
          return names;
        }).catch(error => {
          log.warn(`Error getting ILM stats`, (0, _helpers.withErrorMessage)(error));
          incrementCounter(_types.TelemetryCounter.RUNTIME_ERROR, 'ilm-stats', 1);
          return new Set();
        });

        // 6. Publish ILM policies
        const policyCount = await publishIlmPolicies(ilmNames).then(count => {
          incrementCounter(_types.TelemetryCounter.DOCS_SENT, 'ilm-policies', count);
          return count;
        }).catch(error => {
          log.warn(`Error getting ILM policies`, (0, _helpers.withErrorMessage)(error));
          incrementCounter(_types.TelemetryCounter.RUNTIME_ERROR, 'ilm-policies', 1);
          return 0;
        });

        // 7. Publish index templates
        const indexTemplatesCount = await publishIndexTemplatesStats(indexTemplates.slice(0, taskConfig.indices_threshold)).then(count => {
          incrementCounter(_types.TelemetryCounter.DOCS_SENT, 'index-templates', count);
          return count;
        }).catch(error => {
          log.warn(`Error getting index templates`, (0, _helpers.withErrorMessage)(error));
          incrementCounter(_types.TelemetryCounter.RUNTIME_ERROR, 'index-templates', 1);
          return 0;
        });
        log.info(`Sent EBT events`, {
          datastreams: dsCount,
          indicesSettings: indicesSettingsCount,
          ilms: ilmNames.size,
          indices: indicesCount,
          indexTemplates: indexTemplatesCount,
          policies: policyCount
        });
        await taskMetricsService.end(trace);
        return indicesCount;
      } catch (error) {
        log.warn(`Error running indices metadata task`, (0, _helpers.withErrorMessage)(error));
        await taskMetricsService.end(trace, error);
        return 0;
      }
    }
  };
}