"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createTelemetryDetectionRuleListsTaskConfig = createTelemetryDetectionRuleListsTaskConfig;
var _lodash = require("lodash");
var _constants = require("../constants");
var _helpers = require("../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createTelemetryDetectionRuleListsTaskConfig(maxTelemetryBatch) {
  const taskName = 'Security Solution Detection Rule Lists Telemetry';
  const taskType = 'security:telemetry-detection-rules';
  return {
    type: taskType,
    title: taskName,
    interval: '24h',
    timeout: '10m',
    version: '1.0.0',
    runTask: async (taskId, logger, receiver, sender, taskMetricsService, taskExecutionPeriod) => {
      const mdc = {
        task_id: taskId,
        task_execution_period: taskExecutionPeriod
      };
      const log = (0, _helpers.newTelemetryLogger)(logger.get('detection_rule'), mdc);
      const usageCollector = sender.getTelemetryUsageCluster();
      const usageLabelPrefix = ['security_telemetry', 'detection-rules'];
      const trace = taskMetricsService.start(taskType);
      log.debug('Running telemetry task');
      try {
        const [clusterInfoPromise, licenseInfoPromise] = await Promise.allSettled([receiver.fetchClusterInfo(), receiver.fetchLicenseInfo()]);
        const clusterInfo = (0, _helpers.safeValue)(clusterInfoPromise);
        const licenseInfo = (0, _helpers.safeValue)(licenseInfoPromise);

        // Lists Telemetry: Detection Rules

        const {
          body: prebuiltRules
        } = await receiver.fetchDetectionRules();
        if (!prebuiltRules) {
          log.debug('no prebuilt rules found');
          await taskMetricsService.end(trace);
          return 0;
        }
        const cacheArray = prebuiltRules.hits.hits.reduce((cache, searchHit) => {
          var _searchHit$_source;
          const rule = searchHit._source;
          const ruleId = rule.alert.params.ruleId;
          const shouldNotProcess = rule === null || rule === undefined || ruleId === null || ruleId === undefined || ((_searchHit$_source = searchHit._source) === null || _searchHit$_source === void 0 ? void 0 : _searchHit$_source.alert.params.exceptionsList.length) === 0;
          if (shouldNotProcess) {
            return cache;
          }
          cache.push(rule);
          return cache;
        }, []);
        const detectionRuleExceptions = [];
        for (const item of cacheArray) {
          const ruleVersion = item.alert.params.version;
          for (const ex of item.alert.params.exceptionsList) {
            const listItem = await receiver.fetchDetectionExceptionList(ex.list_id, ruleVersion);
            for (const exceptionItem of listItem.data) {
              detectionRuleExceptions.push(exceptionItem);
            }
          }
        }
        const detectionRuleExceptionsJson = (0, _helpers.templateExceptionList)(detectionRuleExceptions, clusterInfo, licenseInfo, _constants.LIST_DETECTION_RULE_EXCEPTION);
        log.debug('Detection rule exception json length', {
          length: detectionRuleExceptionsJson.length
        });
        usageCollector === null || usageCollector === void 0 ? void 0 : usageCollector.incrementCounter({
          counterName: (0, _helpers.createUsageCounterLabel)(usageLabelPrefix),
          counterType: 'detection_rule_count',
          incrementBy: detectionRuleExceptionsJson.length
        });
        const batches = (0, _helpers.batchTelemetryRecords)((0, _lodash.cloneDeep)(detectionRuleExceptionsJson), maxTelemetryBatch);
        for (const batch of batches) {
          await sender.sendOnDemand(_constants.TELEMETRY_CHANNEL_LISTS, batch);
        }
        await taskMetricsService.end(trace);
        log.debug('Task executed', {
          length: detectionRuleExceptionsJson.length
        });
        return detectionRuleExceptionsJson.length;
      } catch (error) {
        log.warn('Error running detection rule task', (0, _helpers.withErrorMessage)(error));
        await taskMetricsService.end(trace, error);
        return 0;
      }
    }
  };
}