"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ServiceStatus = exports.DEFAULT_RETRY_CONFIG = exports.DEFAULT_QUEUE_CONFIG = exports.AsyncTelemetryEventsSender = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _axios = _interopRequireDefault(require("axios"));
var rx = _interopRequireWildcard(require("rxjs"));
var _lodash = _interopRequireWildcard(require("lodash"));
var _types = require("./types");
var collections = _interopRequireWildcard(require("./collections_helpers"));
var _rxjs_helpers = require("./rxjs_helpers");
var _sender_helpers = require("./sender_helpers");
var _helpers = require("./helpers");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_QUEUE_CONFIG = exports.DEFAULT_QUEUE_CONFIG = {
  bufferTimeSpanMillis: 30 * 1_000,
  inflightEventsThreshold: 1_000,
  maxPayloadSizeBytes: 1024 * 1024 // 1MiB
};
const DEFAULT_RETRY_CONFIG = exports.DEFAULT_RETRY_CONFIG = {
  retryCount: 3,
  retryDelayMillis: 1000
};
class AsyncTelemetryEventsSender {
  constructor(logger) {
    (0, _defineProperty2.default)(this, "retryConfig", void 0);
    (0, _defineProperty2.default)(this, "fallbackQueueConfig", void 0);
    (0, _defineProperty2.default)(this, "queues", void 0);
    (0, _defineProperty2.default)(this, "flush$", new rx.Subject());
    (0, _defineProperty2.default)(this, "events$", new rx.Subject());
    (0, _defineProperty2.default)(this, "finished$", new rx.Subject());
    (0, _defineProperty2.default)(this, "cache", void 0);
    (0, _defineProperty2.default)(this, "status", ServiceStatus.CREATED);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "telemetryReceiver", void 0);
    (0, _defineProperty2.default)(this, "telemetrySetup", void 0);
    (0, _defineProperty2.default)(this, "telemetryUsageCounter", void 0);
    (0, _defineProperty2.default)(this, "senderUtils", void 0);
    (0, _defineProperty2.default)(this, "analytics", void 0);
    this.logger = (0, _helpers.newTelemetryLogger)(logger.get('telemetry_events.async_sender'));
  }
  setup(retryConfig, fallbackQueueConfig, telemetryReceiver, telemetrySetup, telemetryUsageCounter, analytics) {
    this.logger.debug('Setting up service');
    this.ensureStatus(ServiceStatus.CREATED);
    this.retryConfig = retryConfig;
    this.fallbackQueueConfig = fallbackQueueConfig;
    this.queues = new Map();
    this.cache = new _rxjs_helpers.CachedSubject(this.events$);
    this.telemetryReceiver = telemetryReceiver;
    this.telemetrySetup = telemetrySetup;
    this.telemetryUsageCounter = telemetryUsageCounter;
    this.analytics = analytics;
    this.updateStatus(ServiceStatus.CONFIGURED);
  }
  start(telemetryStart) {
    var _this$cache, _this$cache2;
    this.logger.debug('Starting service');
    this.ensureStatus(ServiceStatus.CONFIGURED);
    this.senderUtils = new _sender_helpers.SenderUtils(this.telemetrySetup, telemetryStart, this.telemetryReceiver, this.telemetryUsageCounter);
    (_this$cache = this.cache) === null || _this$cache === void 0 ? void 0 : _this$cache.stop();
    this.events$.pipe(rx.connect(shared$ => {
      const queues$ = Object.values(_types.TelemetryChannel).map(channel => this.queue$(shared$, channel, this.sendEvents.bind(this)));
      return rx.merge(...queues$);
    })).subscribe({
      next: result => {
        if (isFailure(result)) {
          var _this$senderUtils;
          this.logger.warn('Failure! Unable to send events to channel', {
            events: result.events,
            channel: result.channel,
            error_message: result.message
          });
          (_this$senderUtils = this.senderUtils) === null || _this$senderUtils === void 0 ? void 0 : _this$senderUtils.incrementCounter(_types.TelemetryCounter.DOCS_LOST, result.events, result.channel);
        } else {
          var _this$senderUtils2;
          this.logger.debug('Success! events sent to channel', {
            events: result.events,
            channel: result.channel
          });
          (_this$senderUtils2 = this.senderUtils) === null || _this$senderUtils2 === void 0 ? void 0 : _this$senderUtils2.incrementCounter(_types.TelemetryCounter.DOCS_SENT, result.events, result.channel);
        }
      },
      error: error => {
        this.logger.warn('Unexpected error sending events to channel', (0, _helpers.withErrorMessage)(error));
      },
      complete: () => {
        this.logger.debug('Shutting down');
        this.finished$.next();
      }
    });
    (_this$cache2 = this.cache) === null || _this$cache2 === void 0 ? void 0 : _this$cache2.flush();
    this.updateStatus(ServiceStatus.STARTED);
  }
  async stop() {
    var _this$cache3;
    this.logger.debug('Stopping service');
    this.ensureStatus(ServiceStatus.CONFIGURED, ServiceStatus.STARTED);
    const finishPromise = rx.firstValueFrom(this.finished$);
    this.events$.complete();
    (_this$cache3 = this.cache) === null || _this$cache3 === void 0 ? void 0 : _this$cache3.stop();
    await finishPromise;
    this.updateStatus(ServiceStatus.CONFIGURED);
  }
  send(channel, events) {
    this.ensureStatus(ServiceStatus.CONFIGURED, ServiceStatus.STARTED);
    events.forEach(event => {
      this.events$.next({
        channel,
        payload: event
      });
    });
  }
  simulateSend(channel, events) {
    const payloads = [];
    const localEvents$ = rx.of(...events.map(e => {
      return {
        channel,
        payload: e
      };
    }));
    const localSubscription$ = this.queue$(localEvents$, channel, (_ch, p) => {
      const result = {
        events: events.length,
        channel
      };
      payloads.push(...p);
      return Promise.resolve(result);
    }).subscribe();
    localSubscription$.unsubscribe();
    return payloads;
  }
  updateQueueConfig(channel, config) {
    const currentConfig = this.getQueues().get(channel);
    if (!_lodash.default.isEqual(config, currentConfig)) {
      this.getQueues().set(channel, (0, _lodash.cloneDeep)(config));
      // flush the queues to get the new configuration asap
      this.flush$.next();
    }
  }
  updateDefaultQueueConfig(config) {
    if (!_lodash.default.isEqual(config, this.fallbackQueueConfig)) {
      this.fallbackQueueConfig = (0, _lodash.cloneDeep)(config);
      // flush the queues to get the new configuration asap
      this.flush$.next();
    }
  }
  reportEBT(eventTypeOpts, eventData) {
    if (!this.analytics) {
      throw Error('analytics is unavailable');
    }
    this.analytics.reportEvent(eventTypeOpts.eventType, eventData);
  }

  // internal methods
  queue$(upstream$, channel, send) {
    let inflightEventsCounter = 0;
    const inflightEvents$ = new rx.Subject();
    inflightEvents$.subscribe(value => inflightEventsCounter += value);
    return upstream$.pipe(
    // only take events for the configured channel
    rx.filter(event => event.channel === channel), rx.switchMap(event => {
      var _this$senderUtils3;
      if (inflightEventsCounter < this.getConfigFor(channel).inflightEventsThreshold) {
        return rx.of(event);
      }
      this.logger.debug('Dropping event', {
        event,
        channel,
        inflightEventsCounter
      });
      (_this$senderUtils3 = this.senderUtils) === null || _this$senderUtils3 === void 0 ? void 0 : _this$senderUtils3.incrementCounter(_types.TelemetryCounter.DOCS_DROPPED, 1, channel);
      return rx.EMPTY;
    }),
    // update inflight events counter
    rx.tap(() => {
      inflightEvents$.next(1);
    }),
    // buffer events for a while or after a flush$ event is sent (see updateConfig)
    rx.bufferWhen(() => rx.merge(rx.interval(this.getConfigFor(channel).bufferTimeSpanMillis), this.flush$)),
    // exclude empty buffers
    rx.filter(n => n.length > 0),
    // enrich the events
    rx.map(events => events.map(e => this.enrich(e))),
    // serialize the payloads
    rx.map(events => events.map(e => this.serialize(e))),
    // chunk by size
    rx.map(values => collections.chunkedBy(values, this.getConfigFor(channel).maxPayloadSizeBytes, payload => payload.length)), rx.concatAll(),
    // send events to the telemetry server
    rx.concatMap(payloads => (0, _rxjs_helpers.retryOnError$)(this.getRetryConfig().retryCount, this.getRetryConfig().retryDelayMillis, async () => send(channel, payloads))),
    // update inflight events counter
    rx.tap(result => {
      inflightEvents$.next(-result.events);
    }));
  }
  enrich(event) {
    var _this$telemetryReceiv;
    const clusterInfo = (_this$telemetryReceiv = this.telemetryReceiver) === null || _this$telemetryReceiv === void 0 ? void 0 : _this$telemetryReceiv.getClusterInfo();
    if (typeof event.payload === 'object') {
      var _this$telemetryReceiv2;
      let additional = {};
      const licenseInfo = (_this$telemetryReceiv2 = this.telemetryReceiver) === null || _this$telemetryReceiv2 === void 0 ? void 0 : _this$telemetryReceiv2.getLicenseInfo();
      additional = {
        cluster_name: clusterInfo === null || clusterInfo === void 0 ? void 0 : clusterInfo.cluster_name,
        cluster_uuid: clusterInfo === null || clusterInfo === void 0 ? void 0 : clusterInfo.cluster_uuid,
        ...(licenseInfo ? {
          license: (0, _helpers.copyLicenseFields)(licenseInfo)
        } : {})
      };
      event.payload = {
        ...event.payload,
        ...additional
      };
    }
    return event;
  }
  serialize(event) {
    return JSON.stringify(event.payload);
  }
  async sendEvents(channel, events) {
    this.logger.debug('Sending telemetry events to channel', {
      events: events.length,
      channel
    });
    try {
      const senderMetadata = await this.getSenderMetadata(channel);
      const isTelemetryOptedIn = await senderMetadata.isTelemetryOptedIn();
      if (!isTelemetryOptedIn) {
        var _this$senderUtils4;
        (_this$senderUtils4 = this.senderUtils) === null || _this$senderUtils4 === void 0 ? void 0 : _this$senderUtils4.incrementCounter(_types.TelemetryCounter.TELEMETRY_OPTED_OUT, events.length, channel);
        this.logger.warn('Unable to send events to channel: Telemetry is not opted-in.', {
          channel
        });
        throw newFailure('Telemetry is not opted-in', channel, events.length);
      }
      const isElasticTelemetryReachable = await senderMetadata.isTelemetryServicesReachable();
      if (!isElasticTelemetryReachable) {
        var _this$senderUtils5;
        this.logger.warn('Telemetry Services are not reachable.', {
          channel
        });
        (_this$senderUtils5 = this.senderUtils) === null || _this$senderUtils5 === void 0 ? void 0 : _this$senderUtils5.incrementCounter(_types.TelemetryCounter.TELEMETRY_NOT_REACHABLE, events.length, channel);
        throw newFailure('Telemetry Services are not reachable', channel, events.length);
      }
      const body = events.join('\n');
      const telemetryUrl = senderMetadata.telemetryUrl;
      return await _axios.default.post(telemetryUrl, body, {
        headers: {
          ...senderMetadata.telemetryRequestHeaders(),
          'X-Telemetry-Sender': 'async'
        },
        timeout: 10000
      }).then(r => {
        var _this$senderUtils6;
        (_this$senderUtils6 = this.senderUtils) === null || _this$senderUtils6 === void 0 ? void 0 : _this$senderUtils6.incrementCounter(_types.TelemetryCounter.HTTP_STATUS, events.length, channel, r.status.toString());
        if (r.status < 400) {
          return {
            events: events.length,
            channel
          };
        } else {
          this.logger.warn('Unexpected response', {
            status: r.status
          });
          throw newFailure(`Got ${r.status}`, channel, events.length);
        }
      }).catch(error => {
        var _this$senderUtils7;
        (_this$senderUtils7 = this.senderUtils) === null || _this$senderUtils7 === void 0 ? void 0 : _this$senderUtils7.incrementCounter(_types.TelemetryCounter.RUNTIME_ERROR, events.length, channel);
        this.logger.warn('Runtime error', (0, _helpers.withErrorMessage)(error));
        throw newFailure(`Error posting events: ${error}`, channel, events.length);
      });
    } catch (err) {
      if (isFailure(err)) {
        throw err;
      } else {
        var _this$senderUtils8;
        (_this$senderUtils8 = this.senderUtils) === null || _this$senderUtils8 === void 0 ? void 0 : _this$senderUtils8.incrementCounter(_types.TelemetryCounter.FATAL_ERROR, events.length, channel);
        throw newFailure(`Unexpected error posting events: ${err}`, channel, events.length);
      }
    }
  }
  getQueues() {
    if (this.queues === undefined) throw new Error('Service not initialized');
    return this.queues;
  }
  getConfigFor(channel) {
    var _this$queues$get, _this$queues;
    const config = (_this$queues$get = (_this$queues = this.queues) === null || _this$queues === void 0 ? void 0 : _this$queues.get(channel)) !== null && _this$queues$get !== void 0 ? _this$queues$get : this.fallbackQueueConfig;
    if (config === undefined) throw new Error(`No queue config found for channel "${channel}"`);
    return config;
  }
  async getSenderMetadata(channel) {
    var _this$senderUtils9;
    if (this.senderUtils === undefined) throw new Error('Service not initialized');
    return (_this$senderUtils9 = this.senderUtils) === null || _this$senderUtils9 === void 0 ? void 0 : _this$senderUtils9.fetchSenderMetadata(channel);
  }
  getRetryConfig() {
    if (this.retryConfig === undefined) throw new Error('Service not initialized');
    return this.retryConfig;
  }
  ensureStatus(...expected) {
    if (!expected.includes(this.status)) {
      throw new Error(`${this.status}: invalid status. Expected [${expected.join(',')}]`);
    }
  }
  updateStatus(newStatus) {
    this.status = newStatus;
  }
}
exports.AsyncTelemetryEventsSender = AsyncTelemetryEventsSender;
function newFailure(message, channel, events) {
  const failure = {
    name: 'Failure',
    message,
    channel,
    events
  };
  return failure;
}
function isFailure(result) {
  return result !== null && typeof result === 'object' && 'name' in result && 'message' in result && 'events' in result && 'channel' in result;
}
let ServiceStatus = exports.ServiceStatus = /*#__PURE__*/function (ServiceStatus) {
  ServiceStatus["CREATED"] = "CREATED";
  ServiceStatus["CONFIGURED"] = "CONFIGURED";
  ServiceStatus["STARTED"] = "STARTED";
  return ServiceStatus;
}({});