"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.riskEngineCleanupRoute = void 0;
var _utils = require("@kbn/lists-plugin/server/routes/utils");
var _risk_engine_privileges = require("../risk_engine_privileges");
var _constants = require("../../../../../common/constants");
var _audit = require("../audit");
var _audit2 = require("../../audit");
var _translations = require("./translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const riskEngineCleanupRoute = (router, getStartServices) => {
  router.versioned.delete({
    access: 'public',
    path: _constants.RISK_ENGINE_CLEANUP_URL,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution', `${_constants.APP_ID}-entity-analytics`]
      }
    }
  }).addVersion({
    version: _constants.API_VERSIONS.public.v1,
    validate: {}
  }, (0, _risk_engine_privileges.withRiskEnginePrivilegeCheck)(getStartServices, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    const securitySolution = await context.securitySolution;
    const [_, {
      taskManager
    }] = await getStartServices();
    const riskEngineClient = securitySolution.getRiskEngineDataClient();
    const riskScoreDataClient = securitySolution.getRiskScoreDataClient();
    if (!taskManager) {
      var _securitySolution$get;
      (_securitySolution$get = securitySolution.getAuditLogger()) === null || _securitySolution$get === void 0 ? void 0 : _securitySolution$get.log({
        message: 'User attempted to perform a cleanup of risk engine, but the Kibana Task Manager was unavailable',
        event: {
          action: _audit.RiskEngineAuditActions.RISK_ENGINE_REMOVE_TASK,
          category: _audit2.AUDIT_CATEGORY.DATABASE,
          type: _audit2.AUDIT_TYPE.DELETION,
          outcome: _audit2.AUDIT_OUTCOME.FAILURE
        },
        error: {
          message: 'User attempted to perform a cleanup of risk engine, but the Kibana Task Manager was unavailable'
        }
      });
      return siemResponse.error({
        statusCode: 400,
        body: _translations.TASK_MANAGER_UNAVAILABLE_ERROR
      });
    }
    try {
      const errors = await riskEngineClient.tearDown({
        taskManager,
        riskScoreDataClient
      });
      if (errors && errors.length > 0) {
        return siemResponse.error({
          statusCode: errors.some(error => error.message.includes('Risk engine is disabled or deleted already.')) ? 400 : 500,
          body: {
            cleanup_successful: false,
            errors: errors.map((error, seq) => ({
              seq: seq + 1,
              error: error.toString()
            }))
          },
          bypassErrorFormat: true
        });
      } else {
        return response.ok({
          body: {
            cleanup_successful: true
          }
        });
      }
    } catch (error) {
      return siemResponse.error({
        statusCode: 500,
        body: {
          cleanup_successful: false,
          errors: {
            seq: 1,
            error: JSON.stringify(error)
          }
        },
        bypassErrorFormat: true
      });
    }
  }));
};
exports.riskEngineCleanupRoute = riskEngineCleanupRoute;