"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEntityIndexStatus = exports.generateIndexMappings = exports.deleteEntityIndex = exports.createEntityIndex = exports.BASE_ENTITY_INDEX_MAPPING = void 0;
var _entity_analytics = require("../../../../../common/api/entity_analytics");
var _utils = require("../utils");
var _create_or_update_index = require("../../utils/create_or_update_index");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createEntityIndex = async ({
  entityType,
  esClient,
  namespace,
  logger
}) => {
  await (0, _create_or_update_index.createOrUpdateIndex)({
    esClient,
    logger,
    options: {
      index: (0, _utils.getEntitiesIndexName)(entityType, namespace)
    }
  });
};
exports.createEntityIndex = createEntityIndex;
const deleteEntityIndex = ({
  entityType,
  esClient,
  namespace
}) => esClient.indices.delete({
  index: (0, _utils.getEntitiesIndexName)(entityType, namespace)
}, {
  ignore: [404]
});
exports.deleteEntityIndex = deleteEntityIndex;
const getEntityIndexStatus = async ({
  entityType,
  esClient,
  namespace
}) => {
  const index = (0, _utils.getEntitiesIndexName)(entityType, namespace);
  const exists = await esClient.indices.exists({
    index
  }, {
    ignore: [404]
  });
  return {
    id: index,
    installed: exists,
    resource: _entity_analytics.EngineComponentResourceEnum.index
  };
};
exports.getEntityIndexStatus = getEntityIndexStatus;
const generateIndexMappings = description => {
  const identityFieldMappings = {
    [description.identityField]: description.identityFieldMapping
  };
  const otherFieldMappings = description.fields.filter(({
    mapping
  }) => mapping).reduce((acc, {
    destination,
    mapping
  }) => {
    acc[destination] = mapping;
    return acc;
  }, {});
  return {
    properties: {
      ...BASE_ENTITY_INDEX_MAPPING,
      ...identityFieldMappings,
      ...otherFieldMappings
    }
  };
};
exports.generateIndexMappings = generateIndexMappings;
const BASE_ENTITY_INDEX_MAPPING = exports.BASE_ENTITY_INDEX_MAPPING = {
  '@timestamp': {
    type: 'date'
  },
  'asset.criticality': {
    type: 'keyword'
  },
  'entity.name': {
    type: 'keyword'
  },
  'entity.source': {
    type: 'keyword'
  }
};