"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.searchAfterAndBulkCreateFactory = void 0;
var _lodash = require("lodash");
var _single_search_after = require("./single_search_after");
var _filter_events_against_list = require("./large_list_filters/filter_events_against_list");
var _send_telemetry_events = require("./send_telemetry_events");
var _build_events_query = require("./build_events_query");
var _utils = require("./utils");
var _with_security_span = require("../../../../utils/with_security_span");
var i18n = _interopRequireWildcard(require("../translations"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createLoggedRequestsConfig = (isLoggedRequestsEnabled, sortIds, page) => {
  if (!isLoggedRequestsEnabled) {
    return undefined;
  }
  const description = sortIds ? i18n.FIND_EVENTS_AFTER_CURSOR_DESCRIPTION(JSON.stringify(sortIds)) : i18n.FIND_EVENTS_DESCRIPTION;
  return {
    type: 'findDocuments',
    description,
    skipRequestQuery: page > 2 // skipping query logging for performance reasons, so we won't overwhelm Kibana with large response size
  };
};
const searchAfterAndBulkCreateFactory = async ({
  sharedParams,
  enrichment = _lodash.identity,
  eventsTelemetry,
  filter,
  services,
  sortOrder,
  trackTotalHits,
  additionalFilters,
  bulkCreateExecutor,
  getWarningMessage,
  isLoggedRequestsEnabled,
  maxSignalsOverride
}) => {
  const {
    inputIndex: inputIndexPattern,
    runtimeMappings,
    searchAfterSize: pageSize,
    primaryTimestamp,
    secondaryTimestamp,
    unprocessedExceptions: exceptionsList,
    tuple,
    ruleExecutionLogger,
    listClient
  } = sharedParams;
  return (0, _with_security_span.withSecuritySpan)('searchAfterAndBulkCreate', async () => {
    let toReturn = (0, _utils.createSearchAfterReturnType)();
    let searchingIteration = 0;
    const loggedRequests = [];

    // sortId tells us where to start our next consecutive search_after query
    let sortIds;
    const maxSignals = maxSignalsOverride !== null && maxSignalsOverride !== void 0 ? maxSignalsOverride : tuple.maxSignals;
    while (toReturn.createdSignalsCount <= maxSignals) {
      const cycleNum = `cycle ${searchingIteration++}`;
      try {
        var _searchResult$hits$hi;
        ruleExecutionLogger.debug(`[${cycleNum}] Searching events${sortIds ? ` after cursor ${JSON.stringify(sortIds)}` : ''} in index pattern "${inputIndexPattern}"`);
        const searchAfterQuery = (0, _build_events_query.buildEventsSearchQuery)({
          aggregations: undefined,
          index: inputIndexPattern,
          from: tuple.from.toISOString(),
          to: tuple.to.toISOString(),
          runtimeMappings,
          filter,
          size: Math.ceil(Math.min(maxSignals, pageSize)),
          sortOrder,
          searchAfterSortIds: sortIds,
          primaryTimestamp,
          secondaryTimestamp,
          trackTotalHits,
          additionalFilters
        });
        const {
          searchResult,
          searchDuration,
          searchErrors,
          loggedRequests: singleSearchLoggedRequests = []
        } = await (0, _single_search_after.singleSearchAfter)({
          searchRequest: searchAfterQuery,
          services,
          ruleExecutionLogger,
          loggedRequestsConfig: createLoggedRequestsConfig(isLoggedRequestsEnabled, sortIds, searchingIteration)
        });
        toReturn = (0, _utils.mergeReturns)([toReturn, (0, _utils.createSearchAfterReturnTypeFromResponse)({
          searchResult,
          primaryTimestamp
        }), (0, _utils.createSearchAfterReturnType)({
          searchAfterTimes: [searchDuration],
          errors: searchErrors
        })]);
        loggedRequests.push(...singleSearchLoggedRequests);
        // determine if there are any candidate signals to be processed
        const totalHits = (0, _utils.getTotalHitsValue)(searchResult.hits.total);
        const lastSortIds = (0, _utils.getSafeSortIds)((_searchResult$hits$hi = searchResult.hits.hits[searchResult.hits.hits.length - 1]) === null || _searchResult$hits$hi === void 0 ? void 0 : _searchResult$hits$hi.sort);
        if (totalHits === 0 || searchResult.hits.hits.length === 0) {
          ruleExecutionLogger.debug(`[${cycleNum}] Found 0 events ${sortIds ? ` after cursor ${JSON.stringify(sortIds)}` : ''}`);
          break;
        } else {
          ruleExecutionLogger.debug(`[${cycleNum}] Found ${searchResult.hits.hits.length} of total ${totalHits} events${sortIds ? ` after cursor ${JSON.stringify(sortIds)}` : ''}, last cursor ${JSON.stringify(lastSortIds)}`);
        }

        // filter out the search results that match with the values found in the list.
        // the resulting set are signals to be indexed, given they are not duplicates
        // of signals already present in the signals index.
        const [includedEvents, _] = await (0, _filter_events_against_list.filterEventsAgainstList)({
          listClient,
          exceptionsList,
          ruleExecutionLogger,
          events: searchResult.hits.hits
        });

        // only bulk create if there are filteredEvents leftover
        // if there isn't anything after going through the value list filter
        // skip the call to bulk create and proceed to the next search_after,
        // if there is a sort id to continue the search_after with.
        if (includedEvents.length !== 0) {
          const enrichedEvents = await enrichment(includedEvents);
          const bulkCreateResult = await bulkCreateExecutor({
            enrichedEvents,
            toReturn
          });
          ruleExecutionLogger.debug(`[${cycleNum}] Created ${bulkCreateResult.createdItemsCount} alerts from ${enrichedEvents.length} events`);
          (0, _send_telemetry_events.sendAlertTelemetryEvents)(enrichedEvents, bulkCreateResult.createdItems, eventsTelemetry, ruleExecutionLogger);
          if (bulkCreateResult.alertsWereTruncated) {
            toReturn.warningMessages.push(getWarningMessage());
            break;
          }
        }

        // ES can return negative sort id for date field, when sort order set to desc
        // this could happen when event has empty sort field
        // https://github.com/elastic/kibana/issues/174573 (happens to IM rule only since it uses desc order for events search)
        // when negative sort id used in subsequent request it fails, so when negative sort value found we don't do next request
        const hasNegativeNumber = lastSortIds === null || lastSortIds === void 0 ? void 0 : lastSortIds.some(val => val < 0);
        if (lastSortIds != null && lastSortIds.length !== 0 && !hasNegativeNumber) {
          sortIds = lastSortIds;
        } else {
          ruleExecutionLogger.debug(`[${cycleNum}] Unable to fetch last event cursor`);
          break;
        }
      } catch (exc) {
        ruleExecutionLogger.error('Unable to extract/process events or create alerts', JSON.stringify(exc));
        return (0, _utils.mergeReturns)([toReturn, (0, _utils.createSearchAfterReturnType)({
          success: false,
          errors: [`${exc}`]
        })]);
      }
    }
    ruleExecutionLogger.debug(`Completed bulk indexing of ${toReturn.createdSignalsCount} alert`);
    if (isLoggedRequestsEnabled) {
      toReturn.loggedRequests = loggedRequests;
    }
    return toReturn;
  });
};
exports.searchAfterAndBulkCreateFactory = searchAfterAndBulkCreateFactory;