"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEventList = exports.getEventCount = exports.MAX_PER_PAGE = void 0;
var _get_query_filter = require("../../utils/get_query_filter");
var _single_search_after = require("../../utils/single_search_after");
var _build_events_query = require("../../utils/build_events_query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_PER_PAGE = exports.MAX_PER_PAGE = 9000;
const getEventList = async ({
  sharedParams,
  services,
  perPage,
  searchAfter,
  filters,
  eventListConfig,
  indexFields,
  sortOrder = 'desc'
}) => {
  const {
    inputIndex,
    ruleExecutionLogger,
    primaryTimestamp,
    secondaryTimestamp,
    runtimeMappings,
    tuple,
    exceptionFilter,
    completeRule: {
      ruleParams: {
        query,
        language
      }
    }
  } = sharedParams;
  const calculatedPerPage = perPage !== null && perPage !== void 0 ? perPage : MAX_PER_PAGE;
  if (calculatedPerPage > 10000) {
    throw new TypeError('perPage cannot exceed the size of 10000');
  }
  ruleExecutionLogger.debug(`Querying the events items from the index: "${sharedParams.inputIndex}" with searchAfter: "${searchAfter}" for up to ${calculatedPerPage} indicator items`);
  const queryFilter = (0, _get_query_filter.getQueryFilter)({
    query,
    language: language !== null && language !== void 0 ? language : 'kuery',
    filters,
    index: inputIndex,
    exceptionFilter,
    fields: indexFields
  });
  const searchRequest = (0, _build_events_query.buildEventsSearchQuery)({
    aggregations: undefined,
    searchAfterSortIds: searchAfter,
    index: inputIndex,
    from: tuple.from.toISOString(),
    to: tuple.to.toISOString(),
    size: calculatedPerPage,
    filter: queryFilter,
    primaryTimestamp,
    secondaryTimestamp,
    sortOrder,
    trackTotalHits: false,
    runtimeMappings,
    overrideBody: eventListConfig
  });
  const {
    searchResult
  } = await (0, _single_search_after.singleSearchAfter)({
    searchRequest,
    services,
    ruleExecutionLogger
  });
  ruleExecutionLogger.debug(`Retrieved events items of size: ${searchResult.hits.hits.length}`);
  return searchResult;
};

// TODO: possible bug: event count does not respect large value list exceptions, but searchAfterBulkCreate does.
// could lead to worse performance
exports.getEventList = getEventList;
const getEventCount = async ({
  esClient,
  query,
  language,
  filters,
  index,
  tuple,
  primaryTimestamp,
  secondaryTimestamp,
  exceptionFilter,
  indexFields
}) => {
  const queryFilter = (0, _get_query_filter.getQueryFilter)({
    query,
    language: language !== null && language !== void 0 ? language : 'kuery',
    filters,
    index,
    exceptionFilter,
    fields: indexFields
  });
  const eventSearchQueryBodyQuery = (0, _build_events_query.buildEventsSearchQuery)({
    aggregations: undefined,
    index,
    from: tuple.from.toISOString(),
    to: tuple.to.toISOString(),
    filter: queryFilter,
    size: 0,
    primaryTimestamp,
    secondaryTimestamp,
    searchAfterSortIds: undefined,
    runtimeMappings: undefined
  }).query;
  const response = await esClient.count({
    body: {
      query: eventSearchQueryBodyQuery
    },
    ignore_unavailable: true,
    index
  });
  return response.count;
};
exports.getEventCount = getEventCount;