"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateResponseActionsPermissions = exports.validateNonCustomizableUpdateFields = exports.validateNonCustomizablePatchFields = exports.transformValidateBulkError = void 0;
var _lodash = require("lodash");
var _zodHelpers = require("@kbn/zod-helpers");
var _detection_engine = require("../../../../../common/api/detection_engine");
var _constants = require("../../../../../common/endpoint/service/response_actions/constants");
var _custom_http_request_error = require("../../../../utils/custom_http_request_error");
var _rule_schema = require("../../rule_schema");
var _utils = require("../../routes/utils");
var _internal_rule_to_api_response = require("../logic/detection_rules_client/converters/internal_rule_to_api_response");
var _utils2 = require("../logic/detection_rules_client/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const transformValidateBulkError = (ruleId, rule) => {
  if ((0, _rule_schema.hasValidRuleType)(rule)) {
    const transformed = (0, _internal_rule_to_api_response.internalRuleToAPIResponse)(rule);
    const result = _detection_engine.RuleResponse.safeParse(transformed);
    if (!result.success) {
      return (0, _utils.createBulkErrorObject)({
        ruleId,
        statusCode: 500,
        message: (0, _zodHelpers.stringifyZodError)(result.error)
      });
    }
    return result.data;
  } else {
    return (0, _utils.createBulkErrorObject)({
      ruleId,
      statusCode: 500,
      message: 'Internal error transforming'
    });
  }
};

// Temporary functionality until the new System Actions is in place
// for now we want to make sure that user cannot configure Isolate action if has no RBAC permissions to do so
exports.transformValidateBulkError = transformValidateBulkError;
const validateResponseActionsPermissions = async (securitySolution, ruleUpdate, existingRule) => {
  var _ruleUpdate$response_, _existingRule$params, _existingRule$params$, _existingRule$params2;
  if (!rulePayloadContainsResponseActions(ruleUpdate) || existingRule && !ruleObjectContainsResponseActions(existingRule)) {
    return;
  }
  if (((_ruleUpdate$response_ = ruleUpdate.response_actions) === null || _ruleUpdate$response_ === void 0 ? void 0 : _ruleUpdate$response_.length) === 0 && (existingRule === null || existingRule === void 0 ? void 0 : (_existingRule$params = existingRule.params) === null || _existingRule$params === void 0 ? void 0 : (_existingRule$params$ = _existingRule$params.responseActions) === null || _existingRule$params$ === void 0 ? void 0 : _existingRule$params$.length) === 0) {
    return;
  }
  const endpointAuthz = await securitySolution.getEndpointAuthz();

  // finds elements that are not included in both arrays
  const symmetricDifference = (0, _lodash.xorWith)(ruleUpdate.response_actions, existingRule === null || existingRule === void 0 ? void 0 : (_existingRule$params2 = existingRule.params) === null || _existingRule$params2 === void 0 ? void 0 : _existingRule$params2.responseActions, _lodash.isEqual);
  symmetricDifference.forEach(action => {
    if (!('command' in (action === null || action === void 0 ? void 0 : action.params))) {
      return;
    }
    const authzPropName = _constants.RESPONSE_CONSOLE_ACTION_COMMANDS_TO_REQUIRED_AUTHZ[_constants.RESPONSE_ACTION_API_COMMAND_TO_CONSOLE_COMMAND_MAP[action.params.command]];
    const isValid = endpointAuthz[authzPropName];
    if (!isValid) {
      throw new _custom_http_request_error.CustomHttpRequestError(`User is not authorized to change ${action.params.command} response actions`, 403);
    }
  });
};
exports.validateResponseActionsPermissions = validateResponseActionsPermissions;
function rulePayloadContainsResponseActions(rule) {
  return 'response_actions' in rule;
}
function ruleObjectContainsResponseActions(rule) {
  return rule != null && 'params' in rule && 'responseActions' in (rule === null || rule === void 0 ? void 0 : rule.params);
}
const validateNonCustomizableUpdateFields = (ruleUpdate, existingRule) => {
  // We don't allow non-customizable fields to be changed for prebuilt rules
  if (existingRule.rule_source && existingRule.rule_source.type === 'external') {
    if (!(0, _lodash.isEqual)(ruleUpdate.author, existingRule.author)) {
      throw new _utils2.ClientError(`Cannot update "author" field for prebuilt rules`, 400);
    } else if (ruleUpdate.license !== existingRule.license) {
      throw new _utils2.ClientError(`Cannot update "license" field for prebuilt rules`, 400);
    }
  }
};
exports.validateNonCustomizableUpdateFields = validateNonCustomizableUpdateFields;
const validateNonCustomizablePatchFields = (rulePatch, existingRule) => {
  // We don't allow non-customizable fields to be changed for prebuilt rules
  if (existingRule.rule_source && existingRule.rule_source.type === 'external') {
    if (rulePatch.author && !(0, _lodash.isEqual)(rulePatch.author, existingRule.author)) {
      throw new _utils2.ClientError(`Cannot update "author" field for prebuilt rules`, 400);
    } else if (rulePatch.license != null && rulePatch.license !== existingRule.license) {
      throw new _utils2.ClientError(`Cannot update "license" field for prebuilt rules`, 400);
    }
  }
};
exports.validateNonCustomizablePatchFields = validateNonCustomizablePatchFields;