"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createRuleSourceImporter = exports.RuleSourceImporter = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _ensure_latest_rules_package_installed = require("../../../../prebuilt_rules/logic/integrations/ensure_latest_rules_package_installed");
var _calculate_rule_source_for_import = require("../calculate_rule_source_for_import");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Retrieves the rule IDs (`rule_id`s) of available prebuilt rule assets matching those
 * of the specified rules. This information can be used to determine whether
 * the rule being imported is a custom rule or a prebuilt rule.
 *
 * @param rules - A list of {@link RuleSpecifier}s representing the rules being imported.
 * @param ruleAssetsClient - the {@link IPrebuiltRuleAssetsClient} to use for fetching the available rule assets.
 *
 * @returns A list of the prebuilt rule asset IDs that are available.
 *
 */
const fetchAvailableRuleAssetIds = async ({
  rules,
  ruleAssetsClient
}) => {
  const incomingRuleIds = rules.map(rule => rule.rule_id);
  const availableRuleAssetSpecifiers = await ruleAssetsClient.fetchLatestVersions({
    ruleIds: incomingRuleIds
  });
  return availableRuleAssetSpecifiers.map(specifier => specifier.rule_id);
};

/**
 * Retrieves prebuilt rule assets for rules being imported. These
 * assets can be compared to the incoming rules for the purposes of calculating
 * appropriate `rule_source` values.
 *
 * @param rules - A list of {@link RuleSpecifier}s representing the rules being imported.
 *
 * @returns The prebuilt rule assets matching the specified prebuilt
 * rules. Assets match the `rule_id` and `version` of the specified rules.
 * Because of this, there may be less assets returned than specified rules.
 */
const fetchMatchingAssets = async ({
  rules,
  ruleAssetsClient
}) => {
  const incomingRuleVersions = rules.flatMap(rule => {
    if (rule.version == null) {
      return [];
    }
    return {
      rule_id: rule.rule_id,
      version: rule.version
    };
  });
  return ruleAssetsClient.fetchAssetsByVersion(incomingRuleVersions);
};

/**
 *
 * This class contains utilities for assisting with the calculation of
 * `rule_source` during import. It ensures that the system contains the
 * necessary assets, and provides utilities for fetching information from them,
 * necessary for said calculation.
 */
class RuleSourceImporter {
  constructor({
    context,
    prebuiltRuleAssetsClient,
    prebuiltRuleObjectsClient,
    logger
  }) {
    (0, _defineProperty2.default)(this, "context", void 0);
    (0, _defineProperty2.default)(this, "ruleAssetsClient", void 0);
    (0, _defineProperty2.default)(this, "ruleObjectsClient", void 0);
    (0, _defineProperty2.default)(this, "latestPackagesInstalled", false);
    (0, _defineProperty2.default)(this, "matchingAssetsByRuleId", {});
    (0, _defineProperty2.default)(this, "currentRulesById", {});
    (0, _defineProperty2.default)(this, "rulesToImport", []);
    (0, _defineProperty2.default)(this, "availableRuleAssetIds", new Set());
    (0, _defineProperty2.default)(this, "logger", void 0);
    this.ruleAssetsClient = prebuiltRuleAssetsClient;
    this.ruleObjectsClient = prebuiltRuleObjectsClient;
    this.context = context;
    this.logger = logger;
  }

  /**
   *
   * Prepares the importing of rules by ensuring the latest rules
   * package is installed and fetching the associated prebuilt rule assets.
   */
  async setup(rules) {
    if (!this.latestPackagesInstalled) {
      await (0, _ensure_latest_rules_package_installed.ensureLatestRulesPackageInstalled)(this.ruleAssetsClient, this.context, this.logger);
      this.latestPackagesInstalled = true;
    }
    this.rulesToImport = rules.map(rule => ({
      rule_id: rule.rule_id,
      version: rule.version
    }));
    this.matchingAssetsByRuleId = await this.fetchMatchingAssetsByRuleId();
    this.availableRuleAssetIds = new Set(await this.fetchAvailableRuleAssetIds());
    this.currentRulesById = await this.fetchInstalledRulesByIds(this.rulesToImport.map(rule => rule.rule_id));
  }
  isPrebuiltRule(rule) {
    this.validateRuleInput(rule);
    return this.availableRuleAssetIds.has(rule.rule_id);
  }
  calculateRuleSource(rule) {
    this.validateRuleInput(rule);
    return (0, _calculate_rule_source_for_import.calculateRuleSourceForImport)({
      importedRule: rule,
      currentRule: this.currentRulesById[rule.rule_id],
      prebuiltRuleAssetsByRuleId: this.matchingAssetsByRuleId,
      isKnownPrebuiltRule: this.availableRuleAssetIds.has(rule.rule_id)
    });
  }
  async fetchMatchingAssetsByRuleId() {
    this.validateSetupState();
    const matchingAssets = await fetchMatchingAssets({
      rules: this.rulesToImport,
      ruleAssetsClient: this.ruleAssetsClient
    });
    return matchingAssets.reduce((map, asset) => {
      map[asset.rule_id] = asset;
      return map;
    }, {});
  }
  async fetchInstalledRulesByIds(ruleIds) {
    const currentRules = await this.ruleObjectsClient.fetchInstalledRulesByIds({
      ruleIds
    });
    return Object.fromEntries(currentRules.map(rule => [rule.rule_id, rule]));
  }
  async fetchAvailableRuleAssetIds() {
    this.validateSetupState();
    return fetchAvailableRuleAssetIds({
      rules: this.rulesToImport,
      ruleAssetsClient: this.ruleAssetsClient
    });
  }

  /**
   * Runtime sanity checks to ensure no one's calling this stateful instance in the wrong way.
   *  */
  validateSetupState() {
    if (!this.latestPackagesInstalled) {
      throw new Error('Expected rules package to be installed');
    }
  }
  validateRuleInput(rule) {
    if (!this.rulesToImport.some(knownRule => knownRule.rule_id === rule.rule_id && (knownRule.version === rule.version || knownRule.version == null))) {
      throw new Error(`Rule ${rule.rule_id} was not registered during setup.`);
    }
  }
}
exports.RuleSourceImporter = RuleSourceImporter;
const createRuleSourceImporter = ({
  context,
  prebuiltRuleAssetsClient,
  prebuiltRuleObjectsClient,
  logger
}) => {
  return new RuleSourceImporter({
    context,
    prebuiltRuleAssetsClient,
    prebuiltRuleObjectsClient,
    logger
  });
};
exports.createRuleSourceImporter = createRuleSourceImporter;