"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateMlAuth = exports.toggleRuleEnabledOnUpdate = exports.mergeExceptionLists = exports.RuleResponseValidationError = exports.ClientError = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _validation = require("../../../../machine_learning/validation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint-disable max-classes-per-file */

const toggleRuleEnabledOnUpdate = async (rulesClient, existingRule, updatedRule) => {
  if (existingRule.enabled && !updatedRule.enabled) {
    await rulesClient.disableRule({
      id: existingRule.id
    });
    return {
      enabled: false
    };
  }
  if (!existingRule.enabled && updatedRule.enabled) {
    await rulesClient.enableRule({
      id: existingRule.id
    });
    return {
      enabled: true
    };
  }
  return {
    enabled: existingRule.enabled
  };
};
exports.toggleRuleEnabledOnUpdate = toggleRuleEnabledOnUpdate;
const validateMlAuth = async (mlAuthz, ruleType) => {
  (0, _validation.throwAuthzError)(await mlAuthz.validateRuleType(ruleType));
};
exports.validateMlAuth = validateMlAuth;
class ClientError extends Error {
  constructor(message, statusCode) {
    super(message);
    (0, _defineProperty2.default)(this, "statusCode", void 0);
    this.statusCode = statusCode;
  }
}

/**
 * Represents an error that occurred while validating a RuleResponse object.
 * Includes the ruleId (rule signature id) of the rule that failed validation.
 * Thrown when a rule does not match the RuleResponse schema.
 * @param message - The error message
 * @param ruleId - The rule signature id of the rule that failed validation
 * @extends Error
 */
exports.ClientError = ClientError;
class RuleResponseValidationError extends Error {
  constructor({
    message,
    ruleId
  }) {
    super(message);
    (0, _defineProperty2.default)(this, "ruleId", void 0);
    this.ruleId = ruleId;
  }
}

/**
 * Given a rule from the file system and the set of installed rules this will merge the exception lists
 * from the installed rules onto the rules from the file system.
 * @param latestPrebuiltRule The latest prepackaged rule version that might have exceptions_lists
 * @param existingRule The installed rules which might have user driven exceptions_lists
 */
exports.RuleResponseValidationError = RuleResponseValidationError;
const mergeExceptionLists = (latestPrebuiltRule, existingRule) => {
  if (latestPrebuiltRule.exceptions_list != null) {
    if (existingRule.exceptions_list != null) {
      const uniqueExceptionsLists = latestPrebuiltRule.exceptions_list.filter(potentialDuplicateList => existingRule.exceptions_list.every(list => list.list_id !== potentialDuplicateList.list_id));
      return {
        ...latestPrebuiltRule,
        exceptions_list: [...uniqueExceptionsLists, ...existingRule.exceptions_list]
      };
    } else {
      return latestPrebuiltRule;
    }
  } else {
    // Carry over the previous version's exception list
    latestPrebuiltRule.exceptions_list = existingRule.exceptions_list;
    return latestPrebuiltRule;
  }
};
exports.mergeExceptionLists = mergeExceptionLists;