"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkEnableDisableRules = void 0;
var _invariant = require("../../../../../../../common/utils/invariant");
var _validations = require("../../../logic/bulk_actions/validations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bulkEnableDisableRules = async ({
  rules,
  isDryRun,
  rulesClient,
  action: operation,
  mlAuthz
}) => {
  const errors = [];

  // In the first step, we validate if the rules can be enabled
  const validatedRules = [];
  await Promise.all(rules.map(async rule => {
    try {
      await (0, _validations.validateBulkEnableRule)({
        mlAuthz,
        rule
      });
      validatedRules.push(rule);
    } catch (error) {
      errors.push({
        item: rule,
        error
      });
    }
  }));
  if (isDryRun || validatedRules.length === 0) {
    return {
      updatedRules: validatedRules,
      errors
    };
  }

  // Then if it's not a dry run, we enable the rules that passed the validation
  const ruleIds = validatedRules.map(({
    id
  }) => id);

  // Perform actual update using the rulesClient
  const results = operation === 'enable' ? await rulesClient.bulkEnableRules({
    ids: ruleIds
  }) : await rulesClient.bulkDisableRules({
    ids: ruleIds
  });

  // Rule objects returned from the bulkEnableRules are not
  // compatible with the response type. So we need to map them to
  // the original rules and update the enabled field
  errors.push(...results.errors.map(({
    rule: {
      id
    },
    message
  }) => {
    const rule = rules.find(r => r.id === id);
    (0, _invariant.invariant)(rule != null, 'Unexpected rule id');
    return {
      item: rule,
      error: new Error(message)
    };
  }));
  return {
    updatedRules: results.rules,
    errors
  };
};
exports.bulkEnableDisableRules = bulkEnableDisableRules;