"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.excludeLicenseRestrictedRules = excludeLicenseRestrictedRules;
exports.getPossibleUpgrades = getPossibleUpgrades;
exports.rulesToMap = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Converts an array of rules to a Map with rule IDs as keys
 *
 * @param rules Array of rules
 * @returns Map
 */
const rulesToMap = rules => new Map(rules.map(rule => [rule.params.ruleId, rule]));

/**
 * Excludes rules that are not allowed under the current license.
 *
 * @param rules The array of rule objects to filter
 * @param mlAuthz Machine Learning authorization object
 * @returns A new array containing only the rules that are allowed under the current license
 */
exports.rulesToMap = rulesToMap;
async function excludeLicenseRestrictedRules(rules, mlAuthz) {
  const validationResults = await Promise.all(rules.map(rule => mlAuthz.validateRuleType(rule.type)));
  return rules.filter((_rule, index) => validationResults[index].valid);
}
function getUpgradeTargets(currentRules, targetRulesMap) {
  return currentRules.reduce((allUpgradableRules, currentRule) => {
    const targetRule = targetRulesMap.get(currentRule.rule_id);
    if (targetRule && currentRule.version < targetRule.version) {
      allUpgradableRules.push(targetRule);
    }
    return allUpgradableRules;
  }, []);
}

/**
 * Given current and a target rules, returns a list of possible upgrade targets.
 *
 * @param currentRules The list of rules currently installed.
 * @param targetRulesMap A map of the latest available rule versions, with rule_id as the key.
 * @param mlAuthz Machine Learning authorization object
 * @returns An array of target rule version specifiers.
 */
function getPossibleUpgrades(currentRules, targetRulesMap, mlAuthz) {
  const upgradeTargets = getUpgradeTargets(currentRules, targetRulesMap);
  return excludeLicenseRestrictedRules(upgradeTargets, mlAuthz);
}