"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchTagsByVersion = fetchTagsByVersion;
var _invariant = require("../../../../../../../../common/utils/invariant");
var _prebuilt_rule_assets_type = require("../../prebuilt_rule_assets_type");
var _utils = require("../utils");
var _constants = require("../../../../../rule_management/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Fetches unique tags from prebuilt rule assets for specified rule versions.
 *
 * @param savedObjectsClient - Saved Objects client
 * @param versions - An array of rule version specifiers, each containing a rule_id and version
 * @returns A promise that resolves to an array of unique tags, sorted alphabetically.
 */
async function fetchTagsByVersion(savedObjectsClient, versions) {
  if (versions.length === 0) {
    return [];
  }
  const fetchTags = async kqlFilter => {
    var _findResult$aggregati, _findResult$aggregati2;
    const findResult = await savedObjectsClient.find({
      type: _prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE,
      filter: kqlFilter,
      perPage: 0,
      aggs: {
        unique_tags: {
          terms: {
            field: `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.attributes.tags`,
            // "size" parameter is the maximum number of terms returned by the aggregation, default is 10.
            // Setting it to a large number to ensure we get all tags.
            size: _constants.EXPECTED_MAX_TAGS,
            order: {
              _key: 'asc'
            }
          }
        }
      }
    });
    const buckets = ((_findResult$aggregati = findResult.aggregations) === null || _findResult$aggregati === void 0 ? void 0 : (_findResult$aggregati2 = _findResult$aggregati.unique_tags) === null || _findResult$aggregati2 === void 0 ? void 0 : _findResult$aggregati2.buckets) || [];
    (0, _invariant.invariant)(Array.isArray(buckets), 'fetchTagsByVersion: expected buckets to be an array');
    return buckets;
  };
  const filters = (0, _utils.createChunkedFilters)({
    items: versions,
    mapperFn: versionSpecifier => `(${_utils.RULE_ASSET_ATTRIBUTES}.rule_id: ${versionSpecifier.rule_id} AND ${_utils.RULE_ASSET_ATTRIBUTES}.version: ${versionSpecifier.version})`,
    clausesPerItem: 4
  });
  const buckets = await (0, _utils.chunkedFetch)(fetchTags, filters);
  const tags = new Set();
  buckets.forEach(bucket => tags.add(bucket.key));
  return Array.from(tags).sort();
}