"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchLatestVersions = fetchLatestVersions;
var _invariant = require("../../../../../../../../common/utils/invariant");
var _get_existing_prepackaged_rules = require("../../../../../rule_management/logic/search/get_existing_prepackaged_rules");
var _prebuilt_rule_assets_type = require("../../prebuilt_rule_assets_type");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SEVERITY_RANK = {
  low: 20,
  medium: 40,
  high: 60,
  critical: 80
};

/**
 * Fetches the BasicRuleInfo for prebuilt rule assets: rule_id, version and type.
 * By default, fetches BasicRuleInfo for all prebuilt rule assets.
 * If ruleIds are provided, only fetches for the provided ruleIds.
 *
 * @param savedObjectsClient - Saved Objects client
 * @param queryParameters - Optional arguments object
 * @param queryParameters.ruleIds - Optional array of rule IDs to query for
 * @param queryParameters.filter - Optional filter configuration
 * @param queryParameters.sort - Optional sort configuration
 * @returns A promise that resolves to an array of BasicRuleInfo objects (rule_id, version, type).
 */
async function fetchLatestVersions(savedObjectsClient, queryParameters) {
  const {
    ruleIds,
    filter,
    sort
  } = queryParameters !== null && queryParameters !== void 0 ? queryParameters : {};
  if (ruleIds && ruleIds.length === 0) {
    return [];
  }
  const fetchLatestVersionInfo = async kqlFilter => {
    var _findResult$aggregati, _findResult$aggregati2, _findResult$aggregati3;
    const findResult = await savedObjectsClient.find({
      type: _prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE,
      filter: kqlFilter,
      aggs: {
        rules: {
          terms: {
            field: `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.attributes.rule_id`,
            size: _get_existing_prepackaged_rules.MAX_PREBUILT_RULES_COUNT
          },
          aggs: {
            latest_version: {
              top_hits: {
                size: 1,
                sort: [{
                  [`${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.version`]: 'desc'
                }],
                _source: [`${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.rule_id`, `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.version`, `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.type`, `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.name`, `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.tags`, `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.risk_score`, `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.severity`]
              }
            }
          }
        }
      }
    });
    const aggregatedBuckets = (_findResult$aggregati = (_findResult$aggregati2 = findResult.aggregations) === null || _findResult$aggregati2 === void 0 ? void 0 : (_findResult$aggregati3 = _findResult$aggregati2.rules) === null || _findResult$aggregati3 === void 0 ? void 0 : _findResult$aggregati3.buckets) !== null && _findResult$aggregati !== void 0 ? _findResult$aggregati : [];
    (0, _invariant.invariant)(Array.isArray(aggregatedBuckets), 'Expected buckets to be an array');
    return aggregatedBuckets;
  };
  const filters = queryParameters !== null && queryParameters !== void 0 && queryParameters.ruleIds ? (0, _utils.createChunkedFilters)({
    items: queryParameters.ruleIds,
    mapperFn: ruleId => `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.attributes.rule_id: ${ruleId}`,
    clausesPerItem: 2
  }) : undefined;
  const buckets = await (0, _utils.chunkedFetch)(fetchLatestVersionInfo, filters);
  const latestVersions = buckets.map(bucket => {
    const hit = bucket.latest_version.hits.hits[0];
    const soAttributes = hit._source[_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE];
    return {
      rule_id: soAttributes.rule_id,
      version: soAttributes.version,
      type: soAttributes.type,
      name: soAttributes.name,
      tags: soAttributes.tags,
      risk_score: soAttributes.risk_score,
      severity: soAttributes.severity
    };
  });
  const filteredVersions = filterRuleVersions(latestVersions, filter);

  // Since we fetch from ES in chunks, we can't use ES for sorting.
  // So we sort here, once all chunks are fetched.
  return sortRuleVersions(filteredVersions, sort);
}
const filterRuleVersions = (versions, filter) => {
  if (!(filter !== null && filter !== void 0 && filter.fields)) return versions;
  return versions.filter(versionInfo => {
    var _filter$fields$name, _filter$fields$name$i, _filter$fields$tags, _filter$fields$tags$i;
    // Name filter (case-insensitive substring match for all provided terms)
    const nameValues = (_filter$fields$name = filter.fields.name) === null || _filter$fields$name === void 0 ? void 0 : (_filter$fields$name$i = _filter$fields$name.include) === null || _filter$fields$name$i === void 0 ? void 0 : _filter$fields$name$i.values;
    if (nameValues !== null && nameValues !== void 0 && nameValues.length) {
      const nameLower = versionInfo.name.toLowerCase();
      const matchesAllNames = nameValues.every(val => nameLower.includes(val.toLowerCase()));
      if (!matchesAllNames) return false;
    }

    // Tags filter (exact match for all provided tags)
    const tagValues = (_filter$fields$tags = filter.fields.tags) === null || _filter$fields$tags === void 0 ? void 0 : (_filter$fields$tags$i = _filter$fields$tags.include) === null || _filter$fields$tags$i === void 0 ? void 0 : _filter$fields$tags$i.values;
    if (tagValues !== null && tagValues !== void 0 && tagValues.length) {
      const matchesAllTags = tagValues.every(tag => versionInfo.tags.includes(tag));
      if (!matchesAllTags) return false;
    }
    return true;
  });
};
function sortRuleVersions(versions, sort) {
  var _sort$, _sort$0$order, _sort$2;
  const sortField = sort === null || sort === void 0 ? void 0 : (_sort$ = sort[0]) === null || _sort$ === void 0 ? void 0 : _sort$.field;
  if (!sortField) {
    return versions;
  }
  const order = (_sort$0$order = sort === null || sort === void 0 ? void 0 : (_sort$2 = sort[0]) === null || _sort$2 === void 0 ? void 0 : _sort$2.order) !== null && _sort$0$order !== void 0 ? _sort$0$order : 'asc';
  return versions.sort(createRuleVersionInfoCompareFn(sortField, order));
}
function createRuleVersionInfoCompareFn(sortField, order) {
  const direction = order === 'desc' ? -1 : 1;
  switch (sortField) {
    case 'name':
      return (a, b) => a.name.localeCompare(b.name) * direction;
    case 'risk_score':
      return (a, b) => (a.risk_score - b.risk_score) * direction;
    case 'severity':
      return (a, b) => {
        var _SEVERITY_RANK$a$seve, _SEVERITY_RANK$b$seve;
        const rankA = (_SEVERITY_RANK$a$seve = SEVERITY_RANK[a.severity]) !== null && _SEVERITY_RANK$a$seve !== void 0 ? _SEVERITY_RANK$a$seve : -1;
        const rankB = (_SEVERITY_RANK$b$seve = SEVERITY_RANK[b.severity]) !== null && _SEVERITY_RANK$b$seve !== void 0 ? _SEVERITY_RANK$b$seve : -1;
        return (rankA - rankB) * direction;
      };
  }
}