"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchLatestAssets = fetchLatestAssets;
var _get_existing_prepackaged_rules = require("../../../../../rule_management/logic/search/get_existing_prepackaged_rules");
var _prebuilt_rule_assets_type = require("../../prebuilt_rule_assets_type");
var _prebuilt_rule_assets_validation = require("../../prebuilt_rule_assets_validation");
var _invariant = require("../../../../../../../../common/utils/invariant");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Fetches the latest version of each prebuilt rule asset.
 *
 * @param savedObjectsClient - The saved objects client used to query the saved objects store
 * @returns A promise that resolves to an array of prebuilt rule assets.
 */
async function fetchLatestAssets(savedObjectsClient) {
  var _findResult$aggregati, _findResult$aggregati2, _findResult$aggregati3;
  const findResult = await savedObjectsClient.find({
    type: _prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE,
    // Aggregation groups prebuilt rule assets by rule_id and gets a rule with the highest version for each group.
    aggs: {
      rules: {
        terms: {
          field: `${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.attributes.rule_id`,
          size: _get_existing_prepackaged_rules.MAX_PREBUILT_RULES_COUNT
        },
        aggs: {
          latest_version: {
            top_hits: {
              size: 1,
              sort: {
                [`${_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE}.version`]: 'desc'
              }
            }
          }
        }
      }
    }
  });
  const buckets = (_findResult$aggregati = (_findResult$aggregati2 = findResult.aggregations) === null || _findResult$aggregati2 === void 0 ? void 0 : (_findResult$aggregati3 = _findResult$aggregati2.rules) === null || _findResult$aggregati3 === void 0 ? void 0 : _findResult$aggregati3.buckets) !== null && _findResult$aggregati !== void 0 ? _findResult$aggregati : [];
  (0, _invariant.invariant)(Array.isArray(buckets), 'fetchLatestAssets: expected buckets to be an array');
  const ruleAssets = buckets.map(bucket => {
    const hit = bucket.latest_version.hits.hits[0];
    return hit._source[_prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE];
  });
  return (0, _prebuilt_rule_assets_validation.validatePrebuiltRuleAssets)(ruleAssets);
}