"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchAssetsByVersion = fetchAssetsByVersion;
var _get_existing_prepackaged_rules = require("../../../../../rule_management/logic/search/get_existing_prepackaged_rules");
var _prebuilt_rule_assets_type = require("../../prebuilt_rule_assets_type");
var _prebuilt_rule_assets_validation = require("../../prebuilt_rule_assets_validation");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Fetches prebuilt rule assets for specified rule versions.
 *
 * Takes a list of objects with "rule_id" and "version" properties.
 * Returns full prebuilt rule.
 *
 * @param savedObjectsClient - The saved objects client used to query the saved objects store
 * @param versions - An array of rule version specifiers, each containing a rule_id and version.
 * @returns A promise that resolves to an array of prebuilt rule assets.
 */
async function fetchAssetsByVersion(savedObjectsClient, versions) {
  if (versions.length === 0) {
    // NOTE: without early return it would build incorrect filter and fetch all existing saved objects
    return [];
  }
  const filters = (0, _utils.createChunkedFilters)({
    items: versions,
    mapperFn: versionSpecifier => `(${_utils.RULE_ASSET_ATTRIBUTES}.rule_id: ${versionSpecifier.rule_id} AND ${_utils.RULE_ASSET_ATTRIBUTES}.version: ${versionSpecifier.version})`,
    clausesPerItem: 4
  });
  const ruleAssets = await (0, _utils.chunkedFetch)(async filter => {
    // Usage of savedObjectsClient.bulkGet() is ~25% more performant and
    // simplifies deduplication but too many tests get broken.
    // See https://github.com/elastic/kibana/issues/218198
    const findResult = await savedObjectsClient.find({
      type: _prebuilt_rule_assets_type.PREBUILT_RULE_ASSETS_SO_TYPE,
      filter,
      perPage: _get_existing_prepackaged_rules.MAX_PREBUILT_RULES_COUNT
    });
    return findResult.saved_objects.map(so => so.attributes);
  }, filters);

  // Ensure the order of the returned assets matches the order of the "versions" argument.
  const ruleAssetsMap = new Map();
  for (const asset of ruleAssets) {
    ruleAssetsMap.set(asset.rule_id, asset);
  }
  const orderedRuleAssets = versions.map(version => ruleAssetsMap.get(version.rule_id)).filter(asset => asset !== undefined);
  return (0, _prebuilt_rule_assets_validation.validatePrebuiltRuleAssets)(orderedRuleAssets);
}