"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.installEndpointSecurityPrebuiltRule = void 0;
var _create_index_route = require("../../../routes/index/create_index_route");
var _common = require("../../../../../../common");
var _prebuilt_rule_objects_client = require("../rule_objects/prebuilt_rule_objects_client");
var _prebuilt_rule_assets_client = require("../rule_assets/prebuilt_rule_assets_client");
var _create_prebuilt_rules = require("../rule_objects/create_prebuilt_rules");
var _ensure_latest_rules_package_installed = require("./ensure_latest_rules_package_installed");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * As part of a user taking advantage of the Elastic Defend (formerly Endpoint
 * Security) integration from within fleet, we attempt to install the `Endpoint
 * Security (Elastic Defend)` prebuilt rule which will be enabled by default.
 */
const installEndpointSecurityPrebuiltRule = async ({
  logger,
  context,
  request,
  alerts,
  soClient
}) => {
  // Create detection index & rules (if necessary). move past any failure, this is just a convenience
  try {
    await (0, _create_index_route.createDetectionIndex)(context);
  } catch (err) {
    if (err.statusCode !== 409) {
      // 409 -> detection index already exists, which is fine
      logger.warn(`Possible problem creating detection signals index (${err.statusCode}): ${err.message}`);
    }
  }
  try {
    const rulesClient = await alerts.getRulesClientWithRequest(request);
    const detectionRulesClient = context.getDetectionRulesClient();
    const ruleAssetsClient = (0, _prebuilt_rule_assets_client.createPrebuiltRuleAssetsClient)(soClient);
    const ruleObjectsClient = (0, _prebuilt_rule_objects_client.createPrebuiltRuleObjectsClient)(rulesClient);
    const exceptionsListClient = context.getExceptionListClient();
    const elasticDefendRule = await ruleObjectsClient.fetchInstalledRulesByIds({
      ruleIds: [_common.ELASTIC_SECURITY_RULE_ID]
    });
    if (elasticDefendRule.length > 0) {
      // Elastic Defend rule already installed
      return;
    }
    // Elastic Defend rule not installed, find the latest version in the
    // prebuilt rule assets and install it

    // This will create the endpoint list if it does not exist yet
    await (exceptionsListClient === null || exceptionsListClient === void 0 ? void 0 : exceptionsListClient.createEndpointList());

    // Make sure the latest prebuilt rules package is installed (in case the
    // user installs Elastic Defend integration without visiting Security
    // Solution first)
    await (0, _ensure_latest_rules_package_installed.ensureLatestRulesPackageInstalled)(ruleAssetsClient, context, logger);
    const latestRuleVersion = await ruleAssetsClient.fetchLatestVersions({
      ruleIds: [_common.ELASTIC_SECURITY_RULE_ID]
    });
    if (latestRuleVersion.length === 0) {
      logger.error(`Unable to find Elastic Defend rule in the prebuilt rule assets (rule_id: ${_common.ELASTIC_SECURITY_RULE_ID})`);
      return;
    }
    const ruleAssetsToInstall = await ruleAssetsClient.fetchAssetsByVersion(latestRuleVersion);
    await (0, _create_prebuilt_rules.createPrebuiltRules)(detectionRulesClient, ruleAssetsToInstall, logger);
  } catch (err) {
    logger.error(`Unable to create Endpoint Security rule automatically (${err.statusCode}): ${err.message}`);
  }
};
exports.installEndpointSecurityPrebuiltRule = installEndpointSecurityPrebuiltRule;