"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateRuleFieldsDiff = void 0;
var _utility_types = require("../../../../../../../common/utility_types");
var _invariant = require("../../../../../../../common/utils/invariant");
var _three_way_diff = require("../../../../../../../common/api/detection_engine/prebuilt_rules/model/diff/three_way_diff/three_way_diff");
var _diff_calculation_helpers = require("./diff_calculation_helpers");
var _algorithms = require("./algorithms");
var _scalar_array_diff_algorithm = require("./algorithms/scalar_array_diff_algorithm");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const BASE_TYPE_ERROR = `Base version can't be of different rule type`;
const TARGET_TYPE_ERROR = `Target version can't be of different rule type`;

/**
 * Calculates a three-way diff per each top-level rule field.
 * Returns an object which keys are equal to rule's field names and values are
 * three-way diffs calculated for those fields.
 */
const calculateRuleFieldsDiff = (ruleVersions, isRuleCustomized = false) => {
  const commonFieldsDiff = calculateCommonFieldsDiff(ruleVersions, isRuleCustomized);
  const {
    base_version,
    current_version,
    target_version
  } = ruleVersions;
  const hasBaseVersion = base_version !== _three_way_diff.MissingVersion;
  const isRuleTypeDifferentInTargetVersion = current_version.type !== target_version.type;
  const isRuleTypeDifferentInBaseVersion = hasBaseVersion ? current_version.type !== base_version.type : false;
  if (isRuleTypeDifferentInTargetVersion || isRuleTypeDifferentInBaseVersion) {
    // If rule type has been changed by Elastic in the target version (can happen)
    // or by user in the current version (should never happen), we can't calculate the diff
    // only for fields of a single rule type, and need to calculate it for all fields
    // of all the rule types we have.
    // TODO: Try to get rid of "as" casting
    return calculateAllFieldsDiff({
      base_version: base_version,
      current_version: current_version,
      target_version: target_version
    }, isRuleCustomized);
  }
  switch (current_version.type) {
    case 'query':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'query', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'query', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateCustomQueryFieldsDiff({
            base_version,
            current_version,
            target_version
          }, isRuleCustomized)
        };
      }
    case 'saved_query':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'saved_query', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'saved_query', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateSavedQueryFieldsDiff({
            base_version,
            current_version,
            target_version
          }, isRuleCustomized)
        };
      }
    case 'eql':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'eql', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'eql', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateEqlFieldsDiff({
            base_version,
            current_version,
            target_version
          }, isRuleCustomized)
        };
      }
    case 'threat_match':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'threat_match', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'threat_match', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateThreatMatchFieldsDiff({
            base_version,
            current_version,
            target_version
          }, isRuleCustomized)
        };
      }
    case 'threshold':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'threshold', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'threshold', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateThresholdFieldsDiff({
            base_version,
            current_version,
            target_version
          }, isRuleCustomized)
        };
      }
    case 'machine_learning':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'machine_learning', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'machine_learning', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateMachineLearningFieldsDiff({
            base_version,
            current_version,
            target_version
          }, isRuleCustomized)
        };
      }
    case 'new_terms':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'new_terms', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'new_terms', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateNewTermsFieldsDiff({
            base_version,
            current_version,
            target_version
          }, isRuleCustomized)
        };
      }
    case 'esql':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'esql', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'esql', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateEsqlFieldsDiff({
            base_version,
            current_version,
            target_version
          }, isRuleCustomized)
        };
      }
    default:
      {
        return (0, _utility_types.assertUnreachable)(current_version, 'Unhandled rule type');
      }
  }
};
exports.calculateRuleFieldsDiff = calculateRuleFieldsDiff;
const calculateCommonFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, commonFieldsDiffAlgorithms, isRuleCustomized);
};
const commonFieldsDiffAlgorithms = {
  rule_id: _algorithms.simpleDiffAlgorithm,
  /**
   * `version` shouldn't have a conflict. It always get target value automatically.
   * Diff has informational purpose.
   */
  version: _algorithms.forceTargetVersionDiffAlgorithm,
  name: _algorithms.singleLineStringDiffAlgorithm,
  tags: (0, _scalar_array_diff_algorithm.createScalarArrayDiffAlgorithm)({
    missingBaseVersionStrategy: _scalar_array_diff_algorithm.ScalarArrayDiffMissingBaseVersionStrategy.UseTarget
  }),
  description: _algorithms.multiLineStringDiffAlgorithm,
  severity: _algorithms.singleLineStringDiffAlgorithm,
  severity_mapping: _algorithms.simpleDiffAlgorithm,
  risk_score: _algorithms.numberDiffAlgorithm,
  risk_score_mapping: _algorithms.simpleDiffAlgorithm,
  references: (0, _scalar_array_diff_algorithm.createScalarArrayDiffAlgorithm)({
    missingBaseVersionStrategy: _scalar_array_diff_algorithm.ScalarArrayDiffMissingBaseVersionStrategy.UseTarget
  }),
  false_positives: _algorithms.simpleDiffAlgorithm,
  threat: _algorithms.simpleDiffAlgorithm,
  note: _algorithms.multiLineStringDiffAlgorithm,
  setup: _algorithms.multiLineStringDiffAlgorithm,
  related_integrations: _algorithms.simpleDiffAlgorithm,
  required_fields: _algorithms.simpleDiffAlgorithm,
  rule_schedule: _algorithms.simpleDiffAlgorithm,
  max_signals: _algorithms.numberDiffAlgorithm,
  rule_name_override: _algorithms.simpleDiffAlgorithm,
  timestamp_override: _algorithms.simpleDiffAlgorithm,
  timeline_template: _algorithms.simpleDiffAlgorithm,
  building_block: _algorithms.simpleDiffAlgorithm,
  investigation_fields: _algorithms.simpleDiffAlgorithm
};
const calculateCustomQueryFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, customQueryFieldsDiffAlgorithms, isRuleCustomized);
};
const customQueryFieldsDiffAlgorithms = {
  type: _algorithms.ruleTypeDiffAlgorithm,
  kql_query: _algorithms.kqlQueryDiffAlgorithm,
  data_source: _algorithms.dataSourceDiffAlgorithm,
  alert_suppression: _algorithms.simpleDiffAlgorithm
};
const calculateSavedQueryFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, savedQueryFieldsDiffAlgorithms, isRuleCustomized);
};
const savedQueryFieldsDiffAlgorithms = {
  type: _algorithms.ruleTypeDiffAlgorithm,
  kql_query: _algorithms.kqlQueryDiffAlgorithm,
  data_source: _algorithms.dataSourceDiffAlgorithm,
  alert_suppression: _algorithms.simpleDiffAlgorithm
};
const calculateEqlFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, eqlFieldsDiffAlgorithms, isRuleCustomized);
};
const eqlFieldsDiffAlgorithms = {
  type: _algorithms.ruleTypeDiffAlgorithm,
  eql_query: _algorithms.eqlQueryDiffAlgorithm,
  data_source: _algorithms.dataSourceDiffAlgorithm,
  alert_suppression: _algorithms.simpleDiffAlgorithm
};
const calculateEsqlFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, esqlFieldsDiffAlgorithms, isRuleCustomized);
};
const esqlFieldsDiffAlgorithms = {
  type: _algorithms.ruleTypeDiffAlgorithm,
  esql_query: _algorithms.esqlQueryDiffAlgorithm,
  alert_suppression: _algorithms.simpleDiffAlgorithm
};
const calculateThreatMatchFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, threatMatchFieldsDiffAlgorithms, isRuleCustomized);
};
const threatMatchFieldsDiffAlgorithms = {
  type: _algorithms.ruleTypeDiffAlgorithm,
  kql_query: _algorithms.kqlQueryDiffAlgorithm,
  data_source: _algorithms.dataSourceDiffAlgorithm,
  threat_query: _algorithms.kqlQueryDiffAlgorithm,
  threat_index: (0, _scalar_array_diff_algorithm.createScalarArrayDiffAlgorithm)({
    missingBaseVersionStrategy: _scalar_array_diff_algorithm.ScalarArrayDiffMissingBaseVersionStrategy.UseTarget
  }),
  threat_mapping: _algorithms.simpleDiffAlgorithm,
  threat_indicator_path: _algorithms.singleLineStringDiffAlgorithm,
  alert_suppression: _algorithms.simpleDiffAlgorithm
};
const calculateThresholdFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, thresholdFieldsDiffAlgorithms, isRuleCustomized);
};
const thresholdFieldsDiffAlgorithms = {
  type: _algorithms.ruleTypeDiffAlgorithm,
  kql_query: _algorithms.kqlQueryDiffAlgorithm,
  data_source: _algorithms.dataSourceDiffAlgorithm,
  threshold: _algorithms.simpleDiffAlgorithm,
  alert_suppression: _algorithms.simpleDiffAlgorithm
};
const calculateMachineLearningFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, machineLearningFieldsDiffAlgorithms, isRuleCustomized);
};
const machineLearningFieldsDiffAlgorithms = {
  type: _algorithms.ruleTypeDiffAlgorithm,
  machine_learning_job_id: _algorithms.simpleDiffAlgorithm,
  anomaly_threshold: _algorithms.numberDiffAlgorithm,
  alert_suppression: _algorithms.simpleDiffAlgorithm
};
const calculateNewTermsFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, newTermsFieldsDiffAlgorithms, isRuleCustomized);
};
const newTermsFieldsDiffAlgorithms = {
  type: _algorithms.ruleTypeDiffAlgorithm,
  kql_query: _algorithms.kqlQueryDiffAlgorithm,
  data_source: _algorithms.dataSourceDiffAlgorithm,
  new_terms_fields: (0, _scalar_array_diff_algorithm.createScalarArrayDiffAlgorithm)({
    missingBaseVersionStrategy: _scalar_array_diff_algorithm.ScalarArrayDiffMissingBaseVersionStrategy.UseTarget
  }),
  history_window_start: _algorithms.singleLineStringDiffAlgorithm,
  alert_suppression: _algorithms.simpleDiffAlgorithm
};
const calculateAllFieldsDiff = (ruleVersions, isRuleCustomized) => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, allFieldsDiffAlgorithms, isRuleCustomized);
};
const allFieldsDiffAlgorithms = {
  ...commonFieldsDiffAlgorithms,
  ...customQueryFieldsDiffAlgorithms,
  ...savedQueryFieldsDiffAlgorithms,
  ...eqlFieldsDiffAlgorithms,
  ...esqlFieldsDiffAlgorithms,
  ...threatMatchFieldsDiffAlgorithms,
  ...thresholdFieldsDiffAlgorithms,
  ...machineLearningFieldsDiffAlgorithms,
  ...newTermsFieldsDiffAlgorithms,
  type: _algorithms.ruleTypeDiffAlgorithm
};